package online.javadeveloper.devopsadmin.commands;

import online.javadeveloper.devopsadmin.config.EnvironmentHolder;
import online.javadeveloper.devopsadmin.service.ServiceRegistry;
import org.springframework.shell.standard.ShellComponent;
import org.springframework.shell.standard.ShellMethod;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

/**
 * System-Befehle für Status und Informationen.
 */
@ShellComponent
public class SystemCommands {

    private final ServiceRegistry registry;
    private final EnvironmentHolder environmentHolder;
    private final DateTimeFormatter formatter = DateTimeFormatter.ofPattern("dd.MM.yyyy HH:mm:ss");

    public SystemCommands(ServiceRegistry registry, EnvironmentHolder environmentHolder) {
        this.registry = registry;
        this.environmentHolder = environmentHolder;
    }

    @ShellMethod(value = "Zeigt den aktuellen System-Status", key = "status")
    public String status() {
        var now = LocalDateTime.now().format(formatter);
        var serviceCount = registry.getAllServices().size();
        var env = environmentHolder.getCurrent();

        return """
            ╔══════════════════════════════════════════════════╗
            ║           DEVOPS ADMIN SHELL v1.0                ║
            ╠══════════════════════════════════════════════════╣
            ║  Status:       ✓ Online                          ║
            ║  Environment:  %-10s                        ║
            ║  Services:     %-3d registriert                   ║
            ║  Zeitpunkt:    %s                 ║
            ║  Java:         %-10s                        ║
            ╚══════════════════════════════════════════════════╝
            """.formatted(env, serviceCount, now, System.getProperty("java.version"));
    }

    @ShellMethod(value = "Zeigt Systeminformationen", key = "sysinfo")
    public String systemInfo() {
        var runtime = Runtime.getRuntime();
        var maxMemoryMB = runtime.maxMemory() / (1024 * 1024);
        var freeMemoryMB = runtime.freeMemory() / (1024 * 1024);
        var usedMemoryMB = maxMemoryMB - freeMemoryMB;

        return """
            📊 System-Informationen:
            
            OS:              %s %s
            Architektur:     %s
            Prozessoren:     %d verfügbar
            
            JVM Memory:
              Max:           %d MB
              Verwendet:     %d MB
              Frei:          %d MB
            
            Spring Shell:    3.3.3
            Spring Boot:     3.3.6
            """.formatted(
                System.getProperty("os.name"),
                System.getProperty("os.version"),
                System.getProperty("os.arch"),
                runtime.availableProcessors(),
                maxMemoryMB,
                usedMemoryMB,
                freeMemoryMB
        );
    }

    @ShellMethod(value = "Zeigt eine Übersicht aller Commands", key = "commands")
    public String commandOverview() {
        return """
            📋 Command-Übersicht:
            
            SERVICE-VERWALTUNG
              services          Listet alle Services
              service-info      Details zu einem Service
              health-check      Prüft Health (alle oder einer)
              logs              Zeigt Logs eines Services
            
            DEPLOYMENTS
              deploy            Deployed einen Service
              deploy-prod       Production-Deployment (mit Bestätigung)
              rollback          Rollback zur vorherigen Version
              deploy-history    Zeigt Deployment-Verlauf
            
            ENVIRONMENT
              use-env           Wechselt Environment
              current-env       Zeigt aktuelles Environment
              list-envs         Zeigt alle Environments
            
            SYSTEM
              status            System-Status
              sysinfo           System-Informationen
              commands          Diese Übersicht
            
            Für Details: help <command>
            """;
    }
}
