package online.javadeveloper.devopsadmin.service;

import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;

/**
 * Führt Deployments durch und verwaltet den Deployment-Verlauf.
 * 
 * In einer echten Anwendung: Kubernetes API, Docker Registry, etc.
 */
@Service
public class DeploymentService {

    private final ServiceRegistry registry;
    private final List<DeploymentRecord> history = new ArrayList<>();
    private final DateTimeFormatter formatter = DateTimeFormatter.ofPattern("dd.MM.yyyy HH:mm:ss");

    public DeploymentService(ServiceRegistry registry) {
        this.registry = registry;
    }

    public record DeploymentRecord(
            String service,
            String fromVersion,
            String toVersion,
            String environment,
            LocalDateTime timestamp,
            boolean success
    ) {}

    /**
     * Führt ein Deployment durch.
     */
    public String deploy(String serviceName, String version, String environment) {
        if (!registry.exists(serviceName)) {
            return "❌ Service nicht gefunden: " + serviceName;
        }

        var service = registry.get(serviceName).orElseThrow();
        var oldVersion = service.currentVersion();

        var sb = new StringBuilder();
        sb.append("🚀 Deployment: ").append(serviceName).append(":").append(version)
          .append(" → ").append(environment).append("\n\n");

        // Simulierte Deployment-Schritte
        String[] steps = {
                "🔍 Prüfe Image-Verfügbarkeit in Registry...",
                "📦 Lade Image: " + serviceName + ":" + version,
                "🔄 Starte Rolling Update...",
                "⏳ Warte auf Pod-Readiness...",
                "✅ Rollout abgeschlossen!"
        };

        for (int i = 0; i < steps.length; i++) {
            sb.append("  [").append(i + 1).append("/").append(steps.length).append("] ")
              .append(steps[i]).append("\n");
            
            // Simuliere Verzögerung (in echt: echte Operationen)
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }

        // Update Registry
        registry.updateVersion(serviceName, version);

        // Record in History
        history.add(new DeploymentRecord(
                serviceName,
                oldVersion,
                version,
                environment,
                LocalDateTime.now(),
                true
        ));

        sb.append("\n");
        sb.append("━".repeat(45)).append("\n");
        sb.append("✅ Deployment erfolgreich!\n");
        sb.append("   ").append(oldVersion).append(" → ").append(version).append("\n");

        return sb.toString();
    }

    /**
     * Führt einen Rollback durch.
     */
    public String rollback(String serviceName, String environment) {
        if (!registry.exists(serviceName)) {
            return "❌ Service nicht gefunden: " + serviceName;
        }

        // Finde letztes erfolgreiches Deployment
        var lastDeployment = history.stream()
                .filter(d -> d.service().equals(serviceName))
                .filter(d -> d.success())
                .reduce((first, second) -> second);

        if (lastDeployment.isEmpty()) {
            return "⚠️ Kein vorheriges Deployment gefunden für: " + serviceName;
        }

        var previous = lastDeployment.get();
        var currentVersion = registry.get(serviceName).orElseThrow().currentVersion();

        var sb = new StringBuilder();
        sb.append("⏪ Rollback für ").append(serviceName).append(" in ").append(environment).append("\n\n");
        sb.append("  🔍 Suche vorherige Version...\n");
        sb.append("  📌 Gefunden: ").append(previous.fromVersion())
          .append(" (deployed: ").append(previous.timestamp().format(formatter)).append(")\n");
        sb.append("  🔄 Starte Rollback...\n");
        sb.append("  ✅ Rollback abgeschlossen!\n\n");
        sb.append("━".repeat(45)).append("\n");
        sb.append("Aktive Version: ").append(serviceName).append(":").append(previous.fromVersion()).append("\n");

        // Update Registry
        registry.updateVersion(serviceName, previous.fromVersion());

        // Record Rollback
        history.add(new DeploymentRecord(
                serviceName,
                currentVersion,
                previous.fromVersion(),
                environment,
                LocalDateTime.now(),
                true
        ));

        return sb.toString();
    }

    /**
     * Zeigt den Deployment-Verlauf.
     */
    public String getHistory(String serviceName, int limit) {
        var filtered = history.stream()
                .filter(d -> serviceName == null || d.service().equals(serviceName))
                .toList();

        if (filtered.isEmpty()) {
            return "📋 Keine Deployments gefunden.";
        }

        var sb = new StringBuilder();
        sb.append("📋 Deployment-Verlauf");
        if (serviceName != null) {
            sb.append(" für ").append(serviceName);
        }
        sb.append(":\n\n");

        var toShow = filtered.subList(
                Math.max(0, filtered.size() - limit),
                filtered.size()
        );

        for (var record : toShow) {
            var status = record.success() ? "✓" : "✗";
            sb.append("  ").append(status).append(" ")
              .append(record.timestamp().format(formatter)).append(" | ")
              .append(record.service()).append(" ")
              .append(record.fromVersion()).append(" → ").append(record.toVersion())
              .append(" (").append(record.environment()).append(")\n");
        }

        return sb.toString();
    }
}
