package online.javadeveloper.devopsadmin.service;

import org.springframework.stereotype.Service;

import java.util.Random;

/**
 * Prüft die Health von Services.
 * 
 * In einer echten Anwendung würde diese Klasse HTTP-Calls
 * zu den /actuator/health Endpoints machen.
 */
@Service
public class HealthChecker {

    private final ServiceRegistry registry;
    private final Random random = new Random();

    public HealthChecker(ServiceRegistry registry) {
        this.registry = registry;
    }

    /**
     * Prüft die Health aller registrierten Services.
     */
    public String checkAll() {
        var sb = new StringBuilder();
        sb.append("🏥 Health Check Results:\n\n");

        int healthy = 0;
        int unhealthy = 0;

        for (var service : registry.getAllServices()) {
            boolean isHealthy = checkHealth(service.name());
            
            if (isHealthy) {
                sb.append("  ✓ ").append(service.name())
                  .append(" (").append(service.currentVersion()).append(")")
                  .append(": healthy\n");
                healthy++;
            } else {
                sb.append("  ✗ ").append(service.name())
                  .append(" (").append(service.currentVersion()).append(")")
                  .append(": UNHEALTHY\n");
                unhealthy++;
            }
        }

        sb.append("\n");
        sb.append("━".repeat(40)).append("\n");
        sb.append("Zusammenfassung: ")
          .append(healthy).append(" healthy, ")
          .append(unhealthy).append(" unhealthy\n");

        if (unhealthy > 0) {
            sb.append("\n⚠️  Achtung: Es gibt unhealthy Services!");
        } else {
            sb.append("\n✅ Alle Services sind healthy!");
        }

        return sb.toString();
    }

    /**
     * Prüft die Health eines einzelnen Services.
     */
    public String check(String serviceName) {
        if (!registry.exists(serviceName)) {
            return "❌ Service nicht gefunden: " + serviceName;
        }

        var service = registry.get(serviceName).orElseThrow();
        boolean isHealthy = checkHealth(serviceName);
        
        var sb = new StringBuilder();
        sb.append("🏥 Health Check für ").append(serviceName).append("\n\n");
        
        if (isHealthy) {
            sb.append("  Status:    ✓ healthy\n");
            sb.append("  Version:   ").append(service.currentVersion()).append("\n");
            sb.append("  URL:       ").append(service.url()).append("\n");
            sb.append("  Response:  < 50ms\n");
        } else {
            sb.append("  Status:    ✗ UNHEALTHY\n");
            sb.append("  Version:   ").append(service.currentVersion()).append("\n");
            sb.append("  URL:       ").append(service.url()).append("\n");
            sb.append("  Error:     Connection timeout\n");
            sb.append("\n");
            sb.append("  💡 Tipp: Prüfe die Logs mit 'logs ").append(serviceName).append("'\n");
        }

        return sb.toString();
    }

    /**
     * Simuliert einen Health-Check.
     * In einer echten Anwendung: HTTP-Call zu /actuator/health
     */
    private boolean checkHealth(String serviceName) {
        // payment-service ist immer unhealthy (für Demo)
        if ("payment-service".equals(serviceName)) {
            return false;
        }
        // Andere Services: 90% Chance healthy
        return random.nextDouble() > 0.1;
    }
}
