package online.javadeveloper.devopsadmin.service;

import org.springframework.stereotype.Service;

import java.util.*;

/**
 * Verwaltet die Registrierung und Informationen zu allen Services.
 * 
 * In einer echten Anwendung würde diese Klasse mit Kubernetes,
 * Consul oder einer Datenbank kommunizieren.
 */
@Service
public class ServiceRegistry {

    private final Map<String, ServiceInfo> services = new LinkedHashMap<>();

    public ServiceRegistry() {
        // Demo-Services registrieren
        register(new ServiceInfo("user-service", "http://user-service:8080", "staging", "2.1.0"));
        register(new ServiceInfo("order-service", "http://order-service:8080", "staging", "1.8.3"));
        register(new ServiceInfo("payment-service", "http://payment-service:8080", "staging", "3.0.1"));
        register(new ServiceInfo("notification-service", "http://notification-service:8080", "staging", "1.2.0"));
        register(new ServiceInfo("inventory-service", "http://inventory-service:8080", "staging", "2.0.0"));
    }

    public void register(ServiceInfo service) {
        services.put(service.name(), service);
    }

    public boolean exists(String name) {
        return services.containsKey(name);
    }

    public Optional<ServiceInfo> get(String name) {
        return Optional.ofNullable(services.get(name));
    }

    public Collection<ServiceInfo> getAllServices() {
        return Collections.unmodifiableCollection(services.values());
    }

    public Set<String> getAllServiceNames() {
        return Collections.unmodifiableSet(services.keySet());
    }

    public String getLogs(String serviceName, int lines, boolean errorsOnly) {
        if (!exists(serviceName)) {
            return "❌ Service nicht gefunden: " + serviceName;
        }

        // Simulierte Log-Ausgabe
        var sb = new StringBuilder();
        sb.append("📜 Logs für ").append(serviceName).append(" (letzte ")
          .append(lines).append(" Zeilen):\n\n");

        var random = new Random(serviceName.hashCode());
        var levels = errorsOnly 
                ? new String[]{"ERROR", "ERROR", "WARN"} 
                : new String[]{"INFO", "DEBUG", "WARN", "INFO", "ERROR", "INFO"};
        var messages = new String[]{
                "Request processed successfully",
                "Database connection established",
                "Cache miss for key: user_123",
                "Retry attempt 2/3 for external API",
                "Health check passed",
                "Configuration reloaded",
                "Connection timeout after 30s",
                "Invalid request format"
        };

        for (int i = 0; i < Math.min(lines, 15); i++) {
            var level = levels[random.nextInt(levels.length)];
            var message = messages[random.nextInt(messages.length)];
            var time = String.format("2025-01-10 %02d:%02d:%02d",
                    8 + (i / 4), random.nextInt(60), random.nextInt(60));

            sb.append(String.format("  [%s] [%-5s] %s%n", time, level, message));
        }

        return sb.toString();
    }

    public void updateVersion(String serviceName, String newVersion) {
        get(serviceName).ifPresent(info -> {
            services.put(serviceName, new ServiceInfo(
                    info.name(),
                    info.url(),
                    info.environment(),
                    newVersion
            ));
        });
    }
}
