package online.javadeveloper.devopsadmin.commands;

import online.javadeveloper.devopsadmin.service.HealthChecker;
import online.javadeveloper.devopsadmin.service.ServiceRegistry;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.when;

/**
 * Unit-Tests für ServiceCommands.
 * 
 * Diese Tests zeigen, wie Spring Shell Commands getestet werden:
 * - Mocke die Dependencies
 * - Rufe die Command-Methoden direkt auf
 * - Prüfe das Return-Value
 */
@ExtendWith(MockitoExtension.class)
class ServiceCommandsTest {

    @Mock
    private ServiceRegistry registry;

    @Mock
    private HealthChecker healthChecker;

    private ServiceCommands commands;

    @BeforeEach
    void setUp() {
        commands = new ServiceCommands(registry, healthChecker);
    }

    @Nested
    @DisplayName("services command")
    class ServicesCommand {

        @Test
        @DisplayName("sollte Warnung zeigen wenn keine Services registriert")
        void shouldShowWarningWhenNoServices() {
            // Given
            when(registry.getAllServices()).thenReturn(java.util.Collections.emptyList());

            // When
            String result = commands.listServices();

            // Then
            assertThat(result).contains("Keine Services registriert");
        }

        @Test
        @DisplayName("sollte alle Services auflisten")
        void shouldListAllServices() {
            // Given
            var services = java.util.List.of(
                    new online.javadeveloper.devopsadmin.service.ServiceInfo(
                            "user-service", "http://localhost:8080", "staging", "1.0.0"),
                    new online.javadeveloper.devopsadmin.service.ServiceInfo(
                            "order-service", "http://localhost:8081", "staging", "2.0.0")
            );
            when(registry.getAllServices()).thenReturn(services);

            // When
            String result = commands.listServices();

            // Then
            assertThat(result)
                    .contains("user-service")
                    .contains("order-service")
                    .contains("Gesamt: 2 Services");
        }
    }

    @Nested
    @DisplayName("health-check command")
    class HealthCheckCommand {

        @Test
        @DisplayName("sollte alle Services prüfen wenn kein Service angegeben")
        void shouldCheckAllServicesWhenNoServiceSpecified() {
            // Given
            when(healthChecker.checkAll()).thenReturn("All healthy!");

            // When
            String result = commands.healthCheck(null);

            // Then
            assertThat(result).isEqualTo("All healthy!");
        }

        @Test
        @DisplayName("sollte einzelnen Service prüfen wenn angegeben")
        void shouldCheckSingleService() {
            // Given
            when(healthChecker.check("user-service")).thenReturn("user-service: healthy");

            // When
            String result = commands.healthCheck("user-service");

            // Then
            assertThat(result).contains("user-service");
        }
    }

    @Nested
    @DisplayName("logs command")
    class LogsCommand {

        @Test
        @DisplayName("sollte Fehler bei ungültiger Zeilenanzahl")
        void shouldRejectInvalidLineCount() {
            // When
            String result = commands.logs("user-service", 0, false);

            // Then
            assertThat(result).contains("muss zwischen 1 und 100 liegen");
        }

        @Test
        @DisplayName("sollte Fehler bei unbekanntem Service")
        void shouldRejectUnknownService() {
            // Given
            when(registry.exists("unknown")).thenReturn(false);
            when(registry.getAllServiceNames()).thenReturn(java.util.Set.of("user-service"));

            // When
            String result = commands.logs("unknown", 10, false);

            // Then
            assertThat(result)
                    .contains("Service nicht gefunden")
                    .contains("user-service");
        }

        @Test
        @DisplayName("sollte Logs zurückgeben bei gültigen Parametern")
        void shouldReturnLogsForValidService() {
            // Given
            when(registry.exists("user-service")).thenReturn(true);
            when(registry.getLogs("user-service", 10, false))
                    .thenReturn("Logs für user-service");

            // When
            String result = commands.logs("user-service", 10, false);

            // Then
            assertThat(result).isEqualTo("Logs für user-service");
        }
    }
}
