package de.javafleet.collections;

import java.util.Objects;

/**
 * Person MIT korrektem equals() und hashCode()
 * 
 * Diese Klasse zeigt die Best-Practice-Implementierung:
 * - equals() vergleicht Vorname UND Nachname
 * - hashCode() basiert auf denselben Feldern
 * - Beide verwenden Objects.equals() und Objects.hash() für Null-Sicherheit
 * 
 * Mit dieser Implementierung funktionieren HashSet und HashMap korrekt!
 */
public class Person implements Comparable<Person> {
    
    private String vorname;
    private String nachname;
    
    public Person(String vorname, String nachname) {
        this.vorname = vorname;
        this.nachname = nachname;
    }
    
    public String getVorname() {
        return vorname;
    }
    
    public String getNachname() {
        return nachname;
    }
    
    /**
     * equals() - Wann sind zwei Personen "gleich"?
     * 
     * In unserem Fall: Wenn Vorname UND Nachname gleich sind.
     * 
     * Die Implementierung folgt dem Standard-Pattern:
     * 1. Identitätsprüfung (this == o)
     * 2. Null-Check und Typ-Check
     * 3. Cast und Feldvergleich mit Objects.equals()
     */
    @Override
    public boolean equals(Object o) {
        // 1. Identität: Bin ich dasselbe Objekt?
        if (this == o) return true;
        
        // 2. Null-Check und Typ-Check
        if (o == null || getClass() != o.getClass()) return false;
        
        // 3. Cast und Feldvergleich
        Person person = (Person) o;
        return Objects.equals(vorname, person.vorname) 
            && Objects.equals(nachname, person.nachname);
    }
    
    /**
     * hashCode() - Der "Bucket-Index" für HashSet/HashMap
     * 
     * WICHTIG: Muss dieselben Felder verwenden wie equals()!
     * 
     * Wenn a.equals(b) == true, dann MUSS gelten:
     * a.hashCode() == b.hashCode()
     */
    @Override
    public int hashCode() {
        return Objects.hash(vorname, nachname);
    }
    
    /**
     * compareTo() - Für TreeSet/TreeMap Sortierung
     * 
     * Sortiert erst nach Nachname, dann nach Vorname.
     */
    @Override
    public int compareTo(Person other) {
        // Erst nach Nachname
        int nachnameCompare = this.nachname.compareTo(other.nachname);
        if (nachnameCompare != 0) {
            return nachnameCompare;
        }
        // Bei gleichem Nachnamen: nach Vorname
        return this.vorname.compareTo(other.vorname);
    }
    
    @Override
    public String toString() {
        return vorname + " " + nachname;
    }
}
