package de.javafleet.collections;

import java.util.Comparator;
import java.util.Set;
import java.util.TreeSet;

/**
 * Sortierung mit Comparable und Comparator
 * 
 * TreeSet und TreeMap brauchen eine Sortierreihenfolge.
 * Es gibt zwei Wege:
 * 1. Comparable - Die Klasse selbst definiert die Sortierung
 * 2. Comparator - Externe Sortierlogik
 */
public class SortierungDemo {
    
    public static void demo() {
        System.out.println("--- Sortierung mit Comparable ---");
        
        // Person implementiert Comparable<Person>
        // Sortiert nach: Nachname, dann Vorname
        Set<Person> personenNatuerlich = new TreeSet<>();
        personenNatuerlich.add(new Person("Tom", "Müller"));
        personenNatuerlich.add(new Person("Anna", "Schmidt"));
        personenNatuerlich.add(new Person("Max", "Müller"));
        personenNatuerlich.add(new Person("Lisa", "Bauer"));
        
        System.out.println("TreeSet mit Comparable (Nachname, Vorname):");
        for (Person p : personenNatuerlich) {
            System.out.println("  " + p);
        }
        // Lisa Bauer, Max Müller, Tom Müller, Anna Schmidt
        
        System.out.println();
        System.out.println("--- Sortierung mit Comparator ---");
        
        // Eigener Comparator: Nur nach Vorname
        Comparator<Person> nachVorname = (p1, p2) -> 
            p1.getVorname().compareTo(p2.getVorname());
        
        Set<Person> personenNachVorname = new TreeSet<>(nachVorname);
        personenNachVorname.add(new Person("Tom", "Müller"));
        personenNachVorname.add(new Person("Anna", "Schmidt"));
        personenNachVorname.add(new Person("Max", "Müller"));
        personenNachVorname.add(new Person("Lisa", "Bauer"));
        
        System.out.println("TreeSet mit Comparator (nur Vorname):");
        for (Person p : personenNachVorname) {
            System.out.println("  " + p);
        }
        // Anna Schmidt, Lisa Bauer, Max Müller, Tom Müller
        
        System.out.println();
        System.out.println("--- Comparator-Factory-Methoden ---");
        
        // Elegantere Schreibweise mit Comparator.comparing()
        Comparator<Person> nachNachnameReversed = 
            Comparator.comparing(Person::getNachname).reversed();
        
        Set<Person> personenReversed = new TreeSet<>(nachNachnameReversed);
        personenReversed.addAll(personenNatuerlich);
        
        System.out.println("TreeSet mit reversed Comparator:");
        for (Person p : personenReversed) {
            System.out.println("  " + p);
        }
        // Schmidt, Müller (2x), Bauer - umgekehrt!
        
        System.out.println();
        System.out.println("--- Comparator-Chain ---");
        
        // Komplexe Sortierung: Nachname ASC, dann Vorname DESC
        Comparator<Person> komplex = 
            Comparator.comparing(Person::getNachname)
                      .thenComparing(Comparator.comparing(Person::getVorname).reversed());
        
        Set<Person> personenKomplex = new TreeSet<>(komplex);
        personenKomplex.addAll(personenNatuerlich);
        
        System.out.println("Komplexe Sortierung (Nachname ASC, Vorname DESC):");
        for (Person p : personenKomplex) {
            System.out.println("  " + p);
        }
    }
}
