package de.javafleet.generics;

import java.util.List;

/**
 * Demonstration von Bounded Type Parameters.
 * 
 * Bounds schränken ein, welche Typen als Type Argument erlaubt sind.
 */
public class BoundedDemo {
    
    public static void demo() {
        // sum() nur mit Number und Subklassen
        List<Integer> integers = List.of(1, 2, 3, 4, 5);
        List<Double> doubles = List.of(1.5, 2.5, 3.5);
        
        System.out.println("Summe Integers: " + sum(integers));
        System.out.println("Summe Doubles: " + sum(doubles));
        
        // Das würde NICHT kompilieren:
        // List<String> strings = List.of("a", "b");
        // sum(strings);  // ERROR: String extends Number? Nope!
        
        // max() mit Comparable
        System.out.println("Max von 3, 7, 2: " + max(3, 7, 2));
        System.out.println("Max von 3.14, 2.71: " + max(3.14, 2.71));
        System.out.println("Max von 'Apfel', 'Birne': " + max("Apfel", "Birne"));
        
        // NumberBox
        NumberBox<Integer> intBox = new NumberBox<>(42);
        NumberBox<Double> doubleBox = new NumberBox<>(3.14);
        
        System.out.println("NumberBox<Integer>: " + intBox.getAsDouble());
        System.out.println("NumberBox<Double>: " + doubleBox.getAsDouble());
    }
    
    /**
     * Summiert alle Zahlen in einer Liste.
     * 
     * Upper Bound: T muss Number oder eine Subklasse sein.
     * Dadurch können wir doubleValue() aufrufen!
     * 
     * @param <T> Ein Zahlentyp
     * @param zahlen Liste von Zahlen
     * @return Die Summe
     */
    public static <T extends Number> double sum(List<T> zahlen) {
        double summe = 0;
        for (T zahl : zahlen) {
            summe += zahl.doubleValue();  // Geht weil Number.doubleValue() existiert
        }
        return summe;
    }
    
    /**
     * Findet das Maximum von zwei Werten.
     * 
     * Bound: T muss Comparable<T> implementieren (mit sich selbst vergleichbar).
     * 
     * @param <T> Ein vergleichbarer Typ
     * @param a Erster Wert
     * @param b Zweiter Wert
     * @return Der größere Wert
     */
    public static <T extends Comparable<T>> T max(T a, T b) {
        return a.compareTo(b) > 0 ? a : b;
    }
    
    /**
     * Findet das Maximum von drei Werten.
     */
    public static <T extends Comparable<T>> T max(T a, T b, T c) {
        return max(max(a, b), c);
    }
}

/**
 * Eine Box, die nur Zahlen speichern kann.
 * 
 * Beispiel für Upper Bound auf Klassenebene.
 */
class NumberBox<T extends Number> {
    private T value;
    
    public NumberBox(T value) {
        this.value = value;
    }
    
    public T getValue() {
        return value;
    }
    
    /**
     * Gibt den Wert als double zurück.
     * Funktioniert weil T extends Number, und Number hat doubleValue()!
     */
    public double getAsDouble() {
        return value.doubleValue();
    }
    
    public int getAsInt() {
        return value.intValue();
    }
}
