package de.javafleet.generics;

import java.util.ArrayList;
import java.util.List;

/**
 * Demonstration von Wildcards: ?, ? extends, ? super
 * 
 * PECS: Producer Extends, Consumer Super
 */
public class WildcardDemo {
    
    public static void demo() {
        System.out.println("--- Unbounded Wildcard: <?> ---");
        unboundedDemo();
        
        System.out.println();
        System.out.println("--- Upper Bounded: <? extends Number> ---");
        extendsDemo();
        
        System.out.println();
        System.out.println("--- Lower Bounded: <? super Integer> ---");
        superDemo();
        
        System.out.println();
        System.out.println("--- PECS in Aktion: copy() ---");
        pecsDemo();
    }
    
    // ========== UNBOUNDED: <?> ==========
    
    /**
     * Unbounded Wildcard: List<?>
     * - Lesen: nur als Object
     * - Schreiben: nur null
     */
    static void unboundedDemo() {
        List<String> strings = List.of("A", "B", "C");
        List<Integer> integers = List.of(1, 2, 3);
        
        // Beide können an printList() übergeben werden
        System.out.print("Strings: ");
        printList(strings);
        System.out.print("Integers: ");
        printList(integers);
    }
    
    /**
     * Akzeptiert JEDE Liste, egal welchen Typs.
     */
    public static void printList(List<?> liste) {
        for (Object element : liste) {  // Nur als Object lesbar!
            System.out.print(element + " ");
        }
        System.out.println();
        
        // Das geht NICHT:
        // liste.add("Test");  // COMPILE ERROR!
        // liste.add(null);    // Das einzige was geht
    }
    
    // ========== UPPER BOUNDED: <? extends T> ==========
    
    /**
     * Upper Bounded Wildcard: List<? extends Number>
     * - Akzeptiert: List<Number>, List<Integer>, List<Double>, ...
     * - Lesen: als Number
     * - Schreiben: NICHT möglich (außer null)
     */
    static void extendsDemo() {
        List<Integer> integers = List.of(1, 2, 3, 4, 5);
        List<Double> doubles = List.of(1.5, 2.5, 3.5);
        
        System.out.println("Summe Integers: " + sumNumbers(integers));
        System.out.println("Summe Doubles: " + sumNumbers(doubles));
        
        // Das geht NICHT:
        // List<String> strings = List.of("a", "b");
        // sumNumbers(strings);  // ERROR: String extends Number? Nein!
    }
    
    /**
     * Producer: Wir LESEN aus der Liste -> extends
     */
    public static double sumNumbers(List<? extends Number> zahlen) {
        double sum = 0;
        for (Number n : zahlen) {  // Lesen als Number - OK!
            sum += n.doubleValue();
        }
        // zahlen.add(42);  // COMPILE ERROR! Könnte ja List<Double> sein
        return sum;
    }
    
    // ========== LOWER BOUNDED: <? super T> ==========
    
    /**
     * Lower Bounded Wildcard: List<? super Integer>
     * - Akzeptiert: List<Integer>, List<Number>, List<Object>
     * - Lesen: nur als Object
     * - Schreiben: Integer (und Subtypen) möglich
     */
    static void superDemo() {
        List<Integer> integers = new ArrayList<>();
        List<Number> numbers = new ArrayList<>();
        List<Object> objects = new ArrayList<>();
        
        // Alle drei können befüllt werden
        addIntegers(integers);
        addIntegers(numbers);
        addIntegers(objects);
        
        System.out.println("integers: " + integers);
        System.out.println("numbers: " + numbers);
        System.out.println("objects: " + objects);
    }
    
    /**
     * Consumer: Wir SCHREIBEN in die Liste -> super
     */
    public static void addIntegers(List<? super Integer> liste) {
        liste.add(1);
        liste.add(2);
        liste.add(3);
        
        // Lesen geht nur als Object:
        Object obj = liste.get(0);  // OK
        // Integer i = liste.get(0);  // COMPILE ERROR!
    }
    
    // ========== PECS: Producer Extends, Consumer Super ==========
    
    static void pecsDemo() {
        List<Integer> quelle = List.of(1, 2, 3);
        List<Number> ziel = new ArrayList<>();
        
        copy(ziel, quelle);
        
        System.out.println("Quelle (Producer): " + quelle);
        System.out.println("Ziel (Consumer): " + ziel);
    }
    
    /**
     * Kopiert Elemente von src nach dest.
     * 
     * - src: Producer (wir lesen) -> ? extends T
     * - dest: Consumer (wir schreiben) -> ? super T
     * 
     * So kann man z.B. List<Integer> nach List<Number> kopieren!
     */
    public static <T> void copy(List<? super T> dest, List<? extends T> src) {
        for (T element : src) {    // Lesen aus Producer
            dest.add(element);      // Schreiben in Consumer
        }
    }
}
