package de.javafleet.lambdas;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;

/**
 * ✅ LÖSUNG: Lambda Challenges
 */
public class LambdaChallenge {
    
    public static void runChallenges() {
        System.out.println("=== Lambda Challenges - LÖSUNGEN ===");
        System.out.println();
        
        level1();
        level2();
        level3();
    }
    
    static void level1() {
        System.out.println("--- Level 1: Einfache Lambdas ---");
        
        // 1.1: String in Großbuchstaben
        Function<String, String> toUpper = s -> s.toUpperCase();
        
        // 1.2: Prüfen ob mit "A" beginnt
        Predicate<String> startetMitA = s -> s.startsWith("A");
        
        // 1.3: Mit "Hallo " prefixen und ausgeben
        Consumer<String> begruessung = s -> System.out.println("Hallo " + s);
        
        System.out.println("  1.1 toUpper('test'): " + toUpper.apply("test"));
        System.out.println("  1.2 startetMitA('Anna'): " + startetMitA.test("Anna"));
        System.out.println("  1.2 startetMitA('Max'): " + startetMitA.test("Max"));
        System.out.print("  1.3 begruessung: ");
        begruessung.accept("Welt");
        System.out.println();
    }
    
    static void level2() {
        System.out.println("--- Level 2: Sortieren und Filtern ---");
        
        List<String> woerter = new ArrayList<>(List.of(
            "Java", "ist", "eine", "tolle", "Programmiersprache"
        ));
        
        // 2.1: Nach Länge sortieren
        woerter.sort((a, b) -> a.length() - b.length());
        // Oder: woerter.sort(Comparator.comparingInt(String::length));
        System.out.println("  2.1 Nach Länge: " + woerter);
        
        // 2.2: Alphabetisch (case-insensitive)
        woerter.sort(String::compareToIgnoreCase);
        System.out.println("  2.2 Alphabetisch: " + woerter);
        
        // 2.3: Kurze Wörter entfernen
        woerter.removeIf(s -> s.length() < 4);
        System.out.println("  2.3 Ohne kurze Wörter: " + woerter);
        System.out.println();
    }
    
    static void level3() {
        System.out.println("--- Level 3: Method References ---");
        
        List<Produkt> produkte = new ArrayList<>(List.of(
            new Produkt("Laptop", 999.99, "Elektronik"),
            new Produkt("Buch", 19.99, "Medien"),
            new Produkt("Kaffee", 8.99, "Lebensmittel"),
            new Produkt("Smartphone", 799.99, "Elektronik"),
            new Produkt("Tee", 5.99, "Lebensmittel")
        ));
        
        // 3.1: Nach Kategorie, dann nach Preis
        produkte.sort(Comparator
            .comparing(Produkt::getKategorie)
            .thenComparing(Produkt::getPreis));
        
        System.out.println("  3.1 Nach Kategorie und Preis:");
        produkte.forEach(p -> System.out.println("      " + p));
        
        // 3.2: Nach Preis absteigend
        produkte.sort(Comparator.comparing(Produkt::getPreis).reversed());
        
        System.out.println("  3.2 Nach Preis (teuerste zuerst):");
        produkte.forEach(p -> System.out.println("      " + p));
        
        // 3.3: Günstige Produkte (unter 20€)
        System.out.print("  3.3 Günstige Produkte: ");
        produkte.stream()
            .filter(p -> p.getPreis() < 20)
            .map(Produkt::getName)
            .forEach(name -> System.out.print(name + " "));
        System.out.println();
        System.out.println();
    }
}

class Produkt {
    private String name;
    private double preis;
    private String kategorie;
    
    public Produkt(String name, double preis, String kategorie) {
        this.name = name;
        this.preis = preis;
        this.kategorie = kategorie;
    }
    
    public String getName() { return name; }
    public double getPreis() { return preis; }
    public String getKategorie() { return kategorie; }
    
    @Override
    public String toString() {
        return String.format("%s (%.2f€, %s)", name, preis, kategorie);
    }
}
