package de.javafleet.lambdas;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

/**
 * Demonstration von Comparator mit Lambdas und Factory-Methoden.
 */
public class ComparatorDemo {
    
    public static void demo() {
        List<Person> personen = new ArrayList<>(List.of(
            new Person("Anna", "Schmidt", 28),
            new Person("Max", "Müller", 35),
            new Person("Tom", "Schmidt", 22),
            new Person("Lisa", "Weber", 28)
        ));
        
        System.out.println("  Original: " + kurzListe(personen));
        
        // ===== EINFACHES SORTIEREN =====
        
        // Nach Nachname (Lambda)
        personen.sort((a, b) -> a.getNachname().compareTo(b.getNachname()));
        System.out.println("  Nach Nachname (Lambda): " + kurzListe(personen));
        
        // Mit Comparator.comparing (eleganter!)
        personen.sort(Comparator.comparing(Person::getNachname));
        System.out.println("  Nach Nachname (comparing): " + kurzListe(personen));
        
        // ===== CHAINING =====
        
        // Nach Nachname, dann Vorname
        personen.sort(Comparator
            .comparing(Person::getNachname)
            .thenComparing(Person::getVorname));
        System.out.println("  Nach Nachname, dann Vorname: " + kurzListe(personen));
        
        // ===== UMKEHREN =====
        
        // Nach Alter absteigend
        personen.sort(Comparator
            .comparing(Person::getAlter)
            .reversed());
        System.out.println("  Nach Alter (absteigend): " + kurzListe(personen));
        
        // ===== NULL-SAFE =====
        
        List<Person> mitNull = new ArrayList<>(List.of(
            new Person("Anna", "Schmidt", 28),
            new Person("Max", null, 35),  // Nachname ist null!
            new Person("Tom", "Weber", 22)
        ));
        
        // nullsLast: Einträge mit null am Ende
        mitNull.sort(Comparator.comparing(
            Person::getNachname, 
            Comparator.nullsLast(String::compareTo)
        ));
        System.out.println("  Mit null (nullsLast): " + kurzListe(mitNull));
    }
    
    private static String kurzListe(List<Person> liste) {
        return liste.stream()
            .map(p -> p.getVorname() + " " + (p.getNachname() != null ? p.getNachname() : "?"))
            .toList()
            .toString();
    }
}

/**
 * Einfache Person-Klasse für Demos.
 */
class Person {
    private String vorname;
    private String nachname;
    private int alter;
    
    public Person(String vorname, String nachname, int alter) {
        this.vorname = vorname;
        this.nachname = nachname;
        this.alter = alter;
    }
    
    public String getVorname() { return vorname; }
    public String getNachname() { return nachname; }
    public int getAlter() { return alter; }
    
    @Override
    public String toString() {
        return vorname + " " + nachname + " (" + alter + ")";
    }
}
