package de.javafleet.functional;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

/**
 * ✅ LÖSUNG: Functional Interface Challenges
 */
public class FunctionalChallenge {
    
    public static void runChallenges() {
        System.out.println("=== Functional Interface Challenges - LÖSUNGEN ===");
        System.out.println();
        
        level1();
        level2();
        level3();
    }
    
    static void level1() {
        System.out.println("--- Level 1: Basis-Interfaces ---");
        
        // 1.1: String umkehren
        Function<String, String> reverse = s -> new StringBuilder(s).reverse().toString();
        
        // 1.2: Prüfen ob positiv
        Predicate<Integer> istPositiv = n -> n > 0;
        
        // 1.3: Consumer mit Prefix
        Consumer<String> verarbeiter = s -> System.out.println("Verarbeite: " + s);
        
        // 1.4: Zeitstempel Supplier
        Supplier<String> zeitStempel = () -> 
            LocalDateTime.now().format(DateTimeFormatter.ofPattern("HH:mm:ss"));
        
        System.out.println("  1.1 reverse('Hallo'): " + reverse.apply("Hallo"));
        System.out.println("  1.2 istPositiv(5): " + istPositiv.test(5));
        System.out.println("  1.2 istPositiv(-3): " + istPositiv.test(-3));
        System.out.print("  1.3 ");
        verarbeiter.accept("Test");
        System.out.println("  1.4 zeitStempel: " + zeitStempel.get());
        System.out.println();
    }
    
    static void level2() {
        System.out.println("--- Level 2: Komposition ---");
        
        // 2.1: Pipeline
        Function<String, String> normalize = ((Function<String, String>) String::trim)
            .andThen(String::toLowerCase)
            .andThen(s -> s.replace(" ", "_"));
        
        // 2.2: Kombinierte Predicates
        Predicate<String> nichtLeer = s -> !s.isEmpty();
        Predicate<String> keinLeerzeichen = s -> !s.contains(" ");
        Predicate<String> gueltigerUsername = nichtLeer.and(keinLeerzeichen);
        
        // 2.3: Filter
        List<Integer> zahlen = List.of(1, 2, 3, 4, 5, 6, 7, 8, 9, 10);
        List<Integer> geradeZahlen = filter(zahlen, n -> n % 2 == 0);
        
        System.out.println("  2.1 normalize('  Hello World  '): " + 
            normalize.apply("  Hello World  "));
        System.out.println("  2.2 gueltig('max_mueller'): " + 
            gueltigerUsername.test("max_mueller"));
        System.out.println("  2.2 gueltig('max mueller'): " + 
            gueltigerUsername.test("max mueller"));
        System.out.println("  2.2 gueltig(''): " + 
            gueltigerUsername.test(""));
        System.out.println("  2.3 geradeZahlen: " + geradeZahlen);
        System.out.println();
    }
    
    static void level3() {
        System.out.println("--- Level 3: Eigene Interfaces ---");
        
        // 3.1: Map
        List<String> namen = List.of("max", "anna", "tom");
        List<String> upperNamen = map(namen, String::toUpperCase);
        
        // 3.2: ThrowingSupplier - siehe Interface unten
        ThrowingSupplier<String> dateReader = () -> {
            // Könnte IOException werfen in echtem Code
            return "Dateiinhalt";
        };
        try {
            System.out.println("  3.2 ThrowingSupplier: " + dateReader.get());
        } catch (Exception e) {
            System.out.println("  3.2 Fehler: " + e.getMessage());
        }
        
        // 3.3: Multiplikator-Fabrik (Higher-Order Function)
        Function<Integer, Function<Integer, Integer>> multiplikatorFabrik = 
            faktor -> zahl -> zahl * faktor;
        
        Function<Integer, Integer> mal3 = multiplikatorFabrik.apply(3);
        Function<Integer, Integer> mal7 = multiplikatorFabrik.apply(7);
        
        System.out.println("  3.1 upperNamen: " + upperNamen);
        System.out.println("  3.3 mal3.apply(10): " + mal3.apply(10));
        System.out.println("  3.3 mal7.apply(10): " + mal7.apply(10));
        System.out.println();
    }
    
    public static <T> List<T> filter(List<T> liste, Predicate<T> bedingung) {
        List<T> ergebnis = new ArrayList<>();
        for (T element : liste) {
            if (bedingung.test(element)) {
                ergebnis.add(element);
            }
        }
        return ergebnis;
    }
    
    public static <T, R> List<R> map(List<T> liste, Function<T, R> transformation) {
        List<R> ergebnis = new ArrayList<>();
        for (T element : liste) {
            ergebnis.add(transformation.apply(element));
        }
        return ergebnis;
    }
}

@FunctionalInterface
interface ThrowingSupplier<T> {
    T get() throws Exception;
}
