# Tag 6 - BodyTagSupport Examples

**Java Web Aufbau - Tag 6 von 10**  
*Custom Tags mit BodyTagSupport*

---

## 📋 Projekt-Übersicht

Dieses Maven-Projekt enthält **alle lauffähigen Beispiele** aus Tag 6 des Java Web Aufbau Kurses.

**Enthaltene Custom Tags:**
1. **UppercaseBodyTag** - Konvertiert Body zu Uppercase
2. **RepeatTag** - Wiederholt Body n-mal (mit EVAL_BODY_AGAIN)
3. **ForEachTag** - Iteriert über Collections (wie JSTL `<c:forEach>`)
4. **TableTag + ColumnTag** - Nested Tags mit Parent-Child Kommunikation

---

## 🚀 Schnellstart

### Voraussetzungen

- ✅ JDK 21 LTS
- ✅ Apache Maven 3.9+
- ✅ Payara Server 6.x (oder Tomcat 10.x / GlassFish 7.x)
- ✅ NetBeans 22+ (empfohlen)

### Installation & Deployment

**Option 1: Mit NetBeans**

```bash
# 1. Projekt öffnen
File → Open Project → tag06-bodytag-examples auswählen

# 2. Projekt builden
Rechtsklick auf Projekt → Clean and Build

# 3. Deployen
Rechtsklick auf Projekt → Run

# 4. Im Browser öffnen
http://localhost:8080/tag06-bodytag-examples/
```

**Option 2: Command Line**

```bash
# 1. Projekt builden
mvn clean package

# 2. WAR-Datei deployen
# Die generierte Datei liegt in: target/tag06-bodytag-examples.war

# 3. Auf Payara deployen
cp target/tag06-bodytag-examples.war $PAYARA_HOME/glassfish/domains/domain1/autodeploy/

# 4. Im Browser öffnen
http://localhost:8080/tag06-bodytag-examples/
```

**Option 3: Payara Micro (für schnelles Testen)**

```bash
mvn clean package
mvn payara-micro:start
```

---

## 📂 Projekt-Struktur

```
tag06-bodytag-examples/
├── pom.xml                           # Maven Configuration
├── README.md                         # Diese Datei
│
├── src/main/java/
│   └── com/javafleet/
│       ├── tags/                     # Custom Tag Handler
│       │   ├── UppercaseBodyTag.java
│       │   ├── RepeatTag.java
│       │   ├── ForEachTag.java
│       │   ├── TableTag.java
│       │   └── ColumnTag.java
│       ├── model/                    # Model-Klassen (Demo)
│       │   ├── Product.java
│       │   └── User.java
│       └── servlet/                  # Servlets
│           └── DemoServlet.java
│
└── src/main/webapp/
    ├── index.html                    # Landing Page
    └── WEB-INF/
        ├── demo.jsp                  # Demo-Seite (alle Tags)
        └── tlds/
            └── fleet-tags.tld        # Tag Library Descriptor
```

---

## 🎯 Tag Handler Details

### 1. UppercaseBodyTag

**Zweck:** Konvertiert Body-Content zu Uppercase

**Verwendung:**
```jsp
<fleet:uppercaseBody>
    hello world
</fleet:uppercaseBody>
```

**Output:** `HELLO WORLD`

**Technologie:** BodyTagSupport mit `EVAL_BODY_BUFFERED`

---

### 2. RepeatTag

**Zweck:** Wiederholt Body-Content n-mal

**Verwendung:**
```jsp
<fleet:repeat times="5">
    <p>This appears 5 times!</p>
</fleet:repeat>
```

**Attribute:**
- `times` (int, required) - Anzahl Wiederholungen

**Technologie:** `EVAL_BODY_AGAIN` für Iteration

---

### 3. ForEachTag

**Zweck:** Iteriert über Collections (wie JSTL `<c:forEach>`)

**Verwendung:**
```jsp
<fleet:forEach items="${products}" var="product">
    <p>${product.name}: ${product.price}€</p>
</fleet:forEach>
```

**Attribute:**
- `items` (Collection, required) - Die Collection zum Iterieren
- `var` (String, required) - Variablenname für aktuelles Element

**Technologie:** Iterator mit PageContext und `EVAL_BODY_AGAIN`

---

### 4. TableTag + ColumnTag (Nested)

**Zweck:** Generiert HTML-Tabelle aus Collection

**Verwendung:**
```jsp
<fleet:table items="${users}">
    <fleet:column property="name" header="Name" />
    <fleet:column property="email" header="Email" />
    <fleet:column property="age" header="Age" />
</fleet:table>
```

**TableTag Attribute:**
- `items` (Collection, required) - Die Collection für Rows

**ColumnTag Attribute:**
- `property` (String, required) - Property-Name (via Reflection)
- `header` (String, optional) - Header-Text (default: property name)

**Technologie:** Parent-Child Kommunikation mit `findAncestorWithClass()`

---

## 🔧 TLD-Datei

Die Tag Library Descriptor Datei liegt in:
```
src/main/webapp/WEB-INF/tlds/fleet-tags.tld
```

**URI:** `http://javafleet.com/tags`

**In JSP importieren:**
```jsp
<%@ taglib prefix="fleet" uri="http://javafleet.com/tags" %>
```

---

## 📚 Verwendung in eigenen Projekten

### Tag Handler kopieren

1. Kopiere die Tag Handler Klassen aus `src/main/java/com/javafleet/tags/`
2. Kopiere die TLD-Datei nach `src/main/webapp/WEB-INF/tlds/`
3. Passe Package-Namen an (falls nötig)
4. Importiere in JSP: `<%@ taglib prefix="fleet" uri="http://javafleet.com/tags" %>`

### Eigene Tags erstellen

**Basis-Template für BodyTagSupport:**

```java
package com.yourcompany.tags;

import jakarta.servlet.jsp.tagext.BodyTagSupport;
import jakarta.servlet.jsp.tagext.BodyContent;
import jakarta.servlet.jsp.JspException;
import java.io.IOException;

public class YourTag extends BodyTagSupport {
    
    private String yourAttribute;
    
    public void setYourAttribute(String yourAttribute) {
        this.yourAttribute = yourAttribute;
    }
    
    @Override
    public int doStartTag() throws JspException {
        return EVAL_BODY_BUFFERED;
    }
    
    @Override
    public int doAfterBody() throws JspException {
        try {
            BodyContent bodyContent = getBodyContent();
            String content = bodyContent.getString();
            
            // Deine Logik hier
            
            bodyContent.getEnclosingWriter().write(content);
            bodyContent.clearBody();
            
            return SKIP_BODY;  // Oder EVAL_BODY_AGAIN für Iteration
            
        } catch (IOException e) {
            throw new JspException("Error in YourTag", e);
        }
    }
    
    @Override
    public int doEndTag() throws JspException {
        return EVAL_PAGE;
    }
}
```

**TLD-Eintrag:**

```xml
<tag>
    <n>yourTag</n>
    <tag-class>com.yourcompany.tags.YourTag</tag-class>
    <body-content>scriptless</body-content>
    
    <attribute>
        <n>yourAttribute</n>
        <required>true</required>
        <rtexprvalue>true</rtexprvalue>
    </attribute>
</tag>
```

---

## ⚡ Return-Codes Referenz

**In doStartTag():**
- `EVAL_BODY_BUFFERED` (2011) - Body verarbeiten, Buffer erstellen
- `SKIP_BODY` (0) - Body ignorieren

**In doAfterBody():**
- `EVAL_BODY_AGAIN` (2) - Body nochmal verarbeiten (Iteration!)
- `SKIP_BODY` (0) - Fertig, weiter zu doEndTag()

**In doEndTag():**
- `EVAL_PAGE` (6) - JSP normal weitermachen
- `SKIP_PAGE` (5) - Rest der JSP überspringen (selten!)

---

## 🐛 Troubleshooting

**Problem: Tags werden nicht gefunden**

Lösung:
```bash
# 1. Prüfe TLD-Datei liegt in WEB-INF/tlds/
# 2. Prüfe URI in TLD stimmt mit JSP überein
# 3. Server neu starten
# 4. Clean & Build Projekt
```

---

**Problem: IllegalStateException - Cannot buffer body**

Lösung:
```java
// In doStartTag() NICHTS in Output schreiben vor:
return EVAL_BODY_BUFFERED;
```

---

**Problem: Body wird mehrfach ausgegeben**

Lösung:
```java
@Override
public int doAfterBody() throws JspException {
    // ...
    bodyContent.clearBody();  // KRITISCH!
    return EVAL_BODY_AGAIN;
}
```

---

**Problem: NullPointerException bei getBodyContent()**

Lösung:
```java
// MUSS EVAL_BODY_BUFFERED zurückgeben, nicht SKIP_BODY!
@Override
public int doStartTag() throws JspException {
    return EVAL_BODY_BUFFERED;  // Nicht SKIP_BODY!
}
```

---

## 📖 Weitere Ressourcen

**Kurs-Materialien:**
- **Tag 6 Blogpost:** [java-developer.online](https://java-developer.online)
- **GitHub Repository:** [github.com/java-fleet/java-web-aufbau-examples](https://github.com/java-fleet/java-web-aufbau-examples)

**Offizielle Dokumentation:**
- **Jakarta Tags Spec:** https://jakarta.ee/specifications/tags/
- **BodyTagSupport JavaDoc:** https://jakarta.ee/specifications/tags/3.0/apidocs/

**Tutorials:**
- **Baeldung:** https://www.baeldung.com/jsp-custom-tags
- **Oracle Tutorial:** https://docs.oracle.com/javaee/5/tutorial/doc/bnalj.html

---

## 📧 Support

**Fragen? Probleme?**

- **Email:** support@java-developer.online
- **Website:** https://java-developer.online
- **Kurs:** Java Web Aufbau - Tag 6 von 10

---

## 📜 Lizenz

© 2025 Java Fleet Systems Consulting

Dieses Projekt ist Teil des Java Web Aufbau Kurses und steht für Lernzwecke zur Verfügung.

---

**Viel Erfolg mit Custom Tags!** 🚀

*Java Fleet Systems Consulting*
