package com.javafleet.tags;

import jakarta.servlet.jsp.tagext.BodyTagSupport;
import jakarta.servlet.jsp.tagext.BodyContent;
import jakarta.servlet.jsp.JspException;
import java.io.IOException;
import java.util.Collection;
import java.util.Iterator;

/**
 * Iterates over a collection (like JSTL's c:forEach).
 * 
 * Example:
 * <fleet:forEach items="${products}" var="product">
 *     <p>${product.name}: ${product.price}€</p>
 * </fleet:forEach>
 * 
 * @author Java Fleet Systems Consulting
 */
public class ForEachTag extends BodyTagSupport {
    
    // Attribute
    private Collection<?> items;
    private String var;
    
    // Iterator für Collection
    private Iterator<?> iterator;
    
    // Setter
    public void setItems(Collection<?> items) {
        this.items = items;
    }
    
    public void setVar(String var) {
        this.var = var;
    }
    
    @Override
    public int doStartTag() throws JspException {
        // Validierung
        if (items == null || items.isEmpty()) {
            return SKIP_BODY;
        }
        
        // Iterator erstellen
        iterator = items.iterator();
        
        // Erstes Element in PageContext setzen
        if (iterator.hasNext()) {
            Object currentItem = iterator.next();
            pageContext.setAttribute(var, currentItem);
            return EVAL_BODY_BUFFERED;
        } else {
            return SKIP_BODY;
        }
    }
    
    @Override
    public int doAfterBody() throws JspException {
        try {
            // Body-Content ausgeben
            BodyContent bodyContent = getBodyContent();
            String content = bodyContent.getString();
            bodyContent.getEnclosingWriter().write(content);
            
            // Buffer leeren
            bodyContent.clearBody();
            
            // Nächstes Element?
            if (iterator.hasNext()) {
                // Nächstes Element in PageContext setzen
                Object currentItem = iterator.next();
                pageContext.setAttribute(var, currentItem);
                
                // Nochmal iterieren!
                return EVAL_BODY_AGAIN;
            } else {
                // Fertig - Cleanup
                pageContext.removeAttribute(var);
                return SKIP_BODY;
            }
            
        } catch (IOException e) {
            throw new JspException("Error in ForEachTag", e);
        }
    }
    
    @Override
    public int doEndTag() throws JspException {
        // Cleanup
        iterator = null;
        items = null;
        return EVAL_PAGE;
    }
}
