package com.javafleet.tags;

import jakarta.servlet.jsp.tagext.BodyTagSupport;
import jakarta.servlet.jsp.JspException;
import java.io.IOException;
import java.util.Collection;
import java.util.Iterator;
import java.util.ArrayList;
import java.util.List;

/**
 * Generates an HTML table from a collection (Parent tag).
 * Works with ColumnTag child tags.
 * 
 * Example:
 * <fleet:table items="${users}">
 *     <fleet:column property="name" header="Name" />
 *     <fleet:column property="email" header="Email" />
 * </fleet:table>
 * 
 * @author Java Fleet Systems Consulting
 */
public class TableTag extends BodyTagSupport {
    
    private Collection<?> items;
    private Iterator<?> iterator;
    private Object currentItem;
    
    // Columns sammeln von Child-Tags
    private List<ColumnTag> columns = new ArrayList<>();
    
    public void setItems(Collection<?> items) {
        this.items = items;
    }
    
    // Child-Tags registrieren sich hier!
    public void addColumn(ColumnTag column) {
        columns.add(column);
    }
    
    // Child-Tags können aktuelles Item holen
    public Object getCurrentItem() {
        return currentItem;
    }
    
    @Override
    public int doStartTag() throws JspException {
        if (items == null || items.isEmpty()) {
            return SKIP_BODY;
        }
        
        iterator = items.iterator();
        
        // Body erstmal evaluieren, um Columns zu sammeln
        return EVAL_BODY_BUFFERED;
    }
    
    @Override
    public int doAfterBody() throws JspException {
        try {
            // Beim ersten Mal: Table-Header schreiben
            if (iterator.hasNext() && currentItem == null) {
                writeTableHeader();
                currentItem = iterator.next();
                return EVAL_BODY_AGAIN;  // Erste Row
            }
            
            // Row schreiben
            writeTableRow();
            
            // Body-Content verwerfen (wir generieren eigenes HTML)
            getBodyContent().clearBody();
            
            // Nächstes Item?
            if (iterator.hasNext()) {
                currentItem = iterator.next();
                return EVAL_BODY_AGAIN;
            } else {
                writeTableFooter();
                return SKIP_BODY;
            }
            
        } catch (IOException e) {
            throw new JspException("Error in TableTag", e);
        }
    }
    
    private void writeTableHeader() throws IOException {
        StringBuilder html = new StringBuilder();
        html.append("<table class='fleet-table' border='1' cellpadding='8' cellspacing='0'>\n");
        html.append("  <thead>\n");
        html.append("    <tr style='background-color: #667eea; color: white;'>\n");
        
        for (ColumnTag column : columns) {
            html.append("      <th>").append(column.getHeader()).append("</th>\n");
        }
        
        html.append("    </tr>\n");
        html.append("  </thead>\n");
        html.append("  <tbody>\n");
        
        getBodyContent().getEnclosingWriter().write(html.toString());
    }
    
    private void writeTableRow() throws IOException {
        StringBuilder html = new StringBuilder();
        html.append("    <tr>\n");
        
        for (ColumnTag column : columns) {
            String value = column.getValueFromItem(currentItem);
            html.append("      <td>").append(value).append("</td>\n");
        }
        
        html.append("    </tr>\n");
        
        getBodyContent().getEnclosingWriter().write(html.toString());
    }
    
    private void writeTableFooter() throws IOException {
        String html = "  </tbody>\n</table>\n";
        getBodyContent().getEnclosingWriter().write(html);
    }
    
    @Override
    public int doEndTag() throws JspException {
        // Cleanup
        columns.clear();
        iterator = null;
        currentItem = null;
        return EVAL_PAGE;
    }
}
