package de.javafleet.streams;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Demonstration: Collectors.
 */
public class CollectorsDemo {
    
    public static void demo() {
        List<String> namen = List.of("Max", "Anna", "Tom", "Lisa");
        List<Person> personen = Person.getTestPersonen();
        
        // ===== TOLIST, TOSET, TOCOLLECTION =====
        System.out.println("  toList/toSet/toCollection():");
        
        List<String> liste = namen.stream()
            .filter(s -> s.length() > 2)
            .collect(Collectors.toList());
        System.out.println("    toList: " + liste);
        
        Set<String> set = namen.stream()
            .collect(Collectors.toSet());
        System.out.println("    toSet: " + set);
        
        TreeSet<String> treeSet = namen.stream()
            .collect(Collectors.toCollection(TreeSet::new));
        System.out.println("    toCollection(TreeSet): " + treeSet);
        
        // ===== TOMAP =====
        System.out.println("  toMap():");
        
        Map<String, Integer> nameLaenge = namen.stream()
            .collect(Collectors.toMap(
                Function.identity(),
                String::length
            ));
        System.out.println("    Name -> Länge: " + nameLaenge);
        
        Map<String, Person> nameZuPerson = personen.stream()
            .collect(Collectors.toMap(
                Person::getName,
                Function.identity()
            ));
        System.out.println("    Name -> Person: " + nameZuPerson.keySet());
        
        // ===== JOINING =====
        System.out.println("  joining():");
        
        String joined = namen.stream()
            .collect(Collectors.joining());
        System.out.println("    joining(): " + joined);
        
        String mitKomma = namen.stream()
            .collect(Collectors.joining(", "));
        System.out.println("    joining(\", \"): " + mitKomma);
        
        String mitKlammern = namen.stream()
            .collect(Collectors.joining(", ", "[", "]"));
        System.out.println("    joining(\", \", \"[\", \"]\"): " + mitKlammern);
        
        // ===== GROUPINGBY =====
        System.out.println("  groupingBy():");
        
        Map<String, List<Person>> nachAbteilung = personen.stream()
            .collect(Collectors.groupingBy(Person::getAbteilung));
        System.out.println("    Nach Abteilung:");
        nachAbteilung.forEach((abt, list) -> 
            System.out.println("      " + abt + ": " + list));
        
        // Mit Downstream-Collector
        Map<String, Long> anzahlProAbteilung = personen.stream()
            .collect(Collectors.groupingBy(
                Person::getAbteilung,
                Collectors.counting()
            ));
        System.out.println("    Anzahl pro Abteilung: " + anzahlProAbteilung);
        
        Map<String, Double> durchschnittAlter = personen.stream()
            .collect(Collectors.groupingBy(
                Person::getAbteilung,
                Collectors.averagingInt(Person::getAlter)
            ));
        System.out.println("    Ø Alter pro Abteilung: " + durchschnittAlter);
        
        // ===== PARTITIONINGBY =====
        System.out.println("  partitioningBy():");
        
        Map<Boolean, List<Person>> partition = personen.stream()
            .collect(Collectors.partitioningBy(p -> p.getAlter() >= 30));
        System.out.println("    >= 30: " + partition.get(true));
        System.out.println("    < 30: " + partition.get(false));
    }
}
