package de.javafleet.streams;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * ✅ LÖSUNG: Stream Challenges
 */
public class StreamChallenge {
    
    public static void runChallenges() {
        System.out.println("=== Stream Challenges - LÖSUNGEN ===");
        System.out.println();
        
        level1();
        level2();
        level3();
    }
    
    static void level1() {
        System.out.println("--- Level 1: Grundlagen ---");
        
        List<Integer> zahlen = List.of(1, 2, 3, 4, 5, 6, 7, 8, 9, 10);
        List<String> woerter = List.of("Java", "ist", "eine", "tolle", "Sprache");
        
        // 1.1: Filtere > 5 und verdopple
        List<Integer> verdoppelt = zahlen.stream()
            .filter(n -> n > 5)
            .map(n -> n * 2)
            .toList();
        System.out.println("  1.1 Verdoppelt (>5): " + verdoppelt);
        
        // 1.2: Längstes Wort
        Optional<String> laengstes = woerter.stream()
            .max(Comparator.comparing(String::length));
        System.out.println("  1.2 Längstes Wort: " + laengstes.orElse("?"));
        
        // 1.3: Wörter > 3 Zeichen zählen
        long anzahl = woerter.stream()
            .filter(s -> s.length() > 3)
            .count();
        System.out.println("  1.3 Wörter > 3 Zeichen: " + anzahl);
        
        // 1.4: Summe gerader Zahlen
        int summeGerade = zahlen.stream()
            .filter(n -> n % 2 == 0)
            .reduce(0, Integer::sum);
        System.out.println("  1.4 Summe gerade: " + summeGerade);
        
        System.out.println();
    }
    
    static void level2() {
        System.out.println("--- Level 2: Collectors ---");
        
        List<Person> personen = Person.getTestPersonen();
        
        // 2.1: Nach Abteilung gruppieren
        Map<String, List<Person>> nachAbteilung = personen.stream()
            .collect(Collectors.groupingBy(Person::getAbteilung));
        System.out.println("  2.1 Nach Abteilung:");
        nachAbteilung.forEach((k, v) -> System.out.println("      " + k + ": " + v.size()));
        
        // 2.2: Durchschnittsalter pro Abteilung
        Map<String, Double> avgAlter = personen.stream()
            .collect(Collectors.groupingBy(
                Person::getAbteilung,
                Collectors.averagingInt(Person::getAlter)
            ));
        System.out.println("  2.2 Ø Alter: " + avgAlter);
        
        // 2.3: Namen als String
        String namenString = personen.stream()
            .map(Person::getName)
            .collect(Collectors.joining(", "));
        System.out.println("  2.3 Namen: " + namenString);
        
        // 2.4: Name -> Gehalt Map
        Map<String, Double> nameGehalt = personen.stream()
            .collect(Collectors.toMap(
                Person::getName,
                Person::getGehalt
            ));
        System.out.println("  2.4 Gehälter: " + nameGehalt);
        
        System.out.println();
    }
    
    static void level3() {
        System.out.println("--- Level 3: Fortgeschritten ---");
        
        List<Person> personen = Person.getTestPersonen();
        
        // 3.1: Alle Hobbies (flatMap)
        List<String> alleHobbies = personen.stream()
            .flatMap(p -> p.getHobbies().stream())
            .distinct()
            .sorted()
            .toList();
        System.out.println("  3.1 Alle Hobbies: " + alleHobbies);
        
        // 3.2: Bestverdiener pro Abteilung
        Map<String, Optional<Person>> bestverdiener = personen.stream()
            .collect(Collectors.groupingBy(
                Person::getAbteilung,
                Collectors.maxBy(Comparator.comparing(Person::getGehalt))
            ));
        System.out.println("  3.2 Bestverdiener:");
        bestverdiener.forEach((abt, p) -> 
            System.out.println("      " + abt + ": " + p.map(Person::getName).orElse("?")));
        
        // 3.3: Pagination
        List<Person> seite = personen.stream()
            .skip(2)
            .limit(2)
            .toList();
        System.out.println("  3.3 Seite (skip 2, limit 2): " + seite);
        
        // 3.4: Senior/Junior Partition mit Zählung
        Map<Boolean, Long> seniorJunior = personen.stream()
            .collect(Collectors.partitioningBy(
                p -> p.getAlter() >= 30,
                Collectors.counting()
            ));
        System.out.println("  3.4 Senior (>=30): " + seniorJunior.get(true) + 
                          ", Junior (<30): " + seniorJunior.get(false));
        
        System.out.println();
    }
}
