package de.javafleet.fileio;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * ✅ LÖSUNG: File I/O Challenges
 */
public class FileIOChallenge {
    
    public static void runChallenges() {
        System.out.println("=== File I/O Challenges - LÖSUNGEN ===");
        System.out.println();
        
        level1();
        level2();
        level3();
    }
    
    static void level1() {
        System.out.println("--- Level 1: Grundlagen ---");
        
        // 1.1: Zeilen zählen
        Path beispielDatei = Path.of("testdaten/beispiel.txt");
        int zeilenAnzahl = 0;
        try {
            zeilenAnzahl = Files.readAllLines(beispielDatei).size();
        } catch (IOException e) {
            e.printStackTrace();
        }
        System.out.println("  1.1 Zeilen in beispiel.txt: " + zeilenAnzahl);
        
        // 1.2: Datei schreiben
        List<String> zuSchreiben = List.of("Zeile 1", "Zeile 2", "Zeile 3");
        try {
            Files.createDirectories(Path.of("output"));
            Files.write(Path.of("output/challenge1.txt"), zuSchreiben, StandardCharsets.UTF_8);
            System.out.println("  1.2 Datei geschrieben: output/challenge1.txt");
        } catch (IOException e) {
            e.printStackTrace();
        }
        
        // 1.3: Existenz und Größe prüfen
        Path datei = Path.of("testdaten/beispiel.txt");
        boolean existiert = Files.exists(datei);
        long groesse = 0;
        try {
            if (existiert) {
                groesse = Files.size(datei);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        System.out.println("  1.3 Existiert: " + existiert + ", Größe: " + groesse + " bytes");
        
        System.out.println();
    }
    
    static void level2() {
        System.out.println("--- Level 2: Properties ---");
        
        // 2.1: DB-Properties laden und filtern
        Properties config = new Properties();
        try (InputStream input = FileIOChallenge.class.getResourceAsStream("/config.properties")) {
            if (input != null) {
                config.load(input);
                System.out.println("  2.1 DB-Properties:");
                config.stringPropertyNames().stream()
                    .filter(key -> key.startsWith("db."))
                    .sorted()
                    .forEach(key -> System.out.println("    " + key + " = " + config.getProperty(key)));
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        
        // 2.2: Server-Properties erstellen
        try {
            Files.createDirectories(Path.of("output"));
            Properties serverProps = new Properties();
            serverProps.setProperty("server.host", "localhost");
            serverProps.setProperty("server.port", "8080");
            serverProps.setProperty("server.ssl", "true");
            
            try (OutputStream out = Files.newOutputStream(Path.of("output/server.properties"))) {
                serverProps.store(out, "Server-Konfiguration");
            }
            System.out.println("  2.2 Server-Properties erstellt: output/server.properties");
        } catch (IOException e) {
            e.printStackTrace();
        }
        
        // 2.3: Deutsche Nachrichten laden
        Properties messages = new Properties();
        Path messagesPath = Path.of("src/main/resources/messages_de.properties");
        try (Reader reader = Files.newBufferedReader(messagesPath, StandardCharsets.UTF_8)) {
            messages.load(reader);
            System.out.println("  2.3 Deutsche Nachrichten:");
            System.out.println("    app.welcome: " + messages.getProperty("app.welcome"));
            System.out.println("    button.save: " + messages.getProperty("button.save"));
            System.out.println("    error.general: " + messages.getProperty("error.general"));
        } catch (IOException e) {
            e.printStackTrace();
        }
        
        System.out.println();
    }
    
    static void level3() {
        System.out.println("--- Level 3: Fortgeschritten ---");
        
        // 3.1: Alle .java Dateien finden
        System.out.println("  3.1 Java-Dateien im src-Verzeichnis:");
        try (Stream<Path> stream = Files.walk(Path.of("src"))) {
            stream
                .filter(Files::isRegularFile)
                .filter(p -> p.toString().endsWith(".java"))
                .forEach(p -> System.out.println("    " + p));
        } catch (IOException e) {
            e.printStackTrace();
        }
        
        // 3.2: CSV parsen zu Map
        Map<String, String> mitarbeiter = new HashMap<>();
        try (Stream<String> lines = Files.lines(Path.of("testdaten/mitarbeiter.csv"))) {
            mitarbeiter = lines
                .skip(1)  // Header
                .map(line -> line.split(";"))
                .collect(Collectors.toMap(
                    parts -> parts[0],  // Name
                    parts -> parts[2]   // Abteilung
                ));
        } catch (IOException e) {
            e.printStackTrace();
        }
        System.out.println("  3.2 Mitarbeiter: " + mitarbeiter);
        
        // 3.3: Einfacher Config-Manager
        System.out.println("  3.3 Config-Manager:");
        Path configPath = Path.of("output/app-config.properties");
        Properties appConfig = loadOrCreateConfig(configPath);
        System.out.println("    Geladene Config: " + appConfig.getProperty("app.initialized"));
        
        // Wert ändern und speichern
        appConfig.setProperty("app.last.access", java.time.LocalDateTime.now().toString());
        saveConfig(configPath, appConfig);
        System.out.println("    Config gespeichert mit neuem Zeitstempel");
        
        System.out.println();
    }
    
    /**
     * Lädt Properties aus Datei oder erstellt Default-Werte.
     */
    private static Properties loadOrCreateConfig(Path path) {
        Properties props = new Properties();
        
        if (Files.exists(path)) {
            try (InputStream in = Files.newInputStream(path)) {
                props.load(in);
            } catch (IOException e) {
                e.printStackTrace();
            }
        } else {
            // Default-Werte setzen
            props.setProperty("app.name", "MyApp");
            props.setProperty("app.version", "1.0.0");
            props.setProperty("app.initialized", java.time.LocalDateTime.now().toString());
            saveConfig(path, props);
        }
        
        return props;
    }
    
    /**
     * Speichert Properties in Datei.
     */
    private static void saveConfig(Path path, Properties props) {
        try {
            Files.createDirectories(path.getParent());
            try (OutputStream out = Files.newOutputStream(path)) {
                props.store(out, "Application Configuration");
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }
}
