package de.javafleet.fileio;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.LocalDateTime;
import java.util.Properties;

/**
 * Demonstration: Properties-Dateien laden und speichern.
 * 
 * Properties sind DER Standard für Konfigurationsdateien in Java!
 */
public class PropertiesDemo {
    
    public static void demo() {
        // ===== AUS CLASSPATH LADEN (für JARs!) =====
        System.out.println("  Properties aus Classpath laden:");
        Properties config = new Properties();
        try (InputStream input = PropertiesDemo.class.getResourceAsStream("/config.properties")) {
            if (input != null) {
                config.load(input);
                System.out.println("    Geladen aus Classpath!");
                System.out.println("    app.name: " + config.getProperty("app.name"));
                System.out.println("    db.host: " + config.getProperty("db.host"));
                System.out.println("    db.port: " + config.getProperty("db.port"));
            } else {
                System.out.println("    config.properties nicht im Classpath gefunden!");
            }
        } catch (IOException e) {
            System.out.println("    Fehler: " + e.getMessage());
        }
        
        // ===== MIT DEFAULT-WERTEN =====
        System.out.println("  Properties mit Default-Werten:");
        String debug = config.getProperty("app.debug", "false");
        String timeout = config.getProperty("db.timeout", "30");  // Existiert nicht -> Default
        int maxConn = Integer.parseInt(config.getProperty("app.max.connections", "50"));
        boolean debugMode = Boolean.parseBoolean(config.getProperty("app.debug", "false"));
        
        System.out.println("    app.debug: " + debug);
        System.out.println("    db.timeout (mit Default): " + timeout);
        System.out.println("    app.max.connections (als int): " + maxConn);
        System.out.println("    debug als boolean: " + debugMode);
        
        // ===== ALLE DB-PROPERTIES FILTERN =====
        System.out.println("  Alle 'db.*' Properties:");
        config.stringPropertyNames().stream()
            .filter(key -> key.startsWith("db."))
            .sorted()
            .forEach(key -> System.out.println("    " + key + " = " + config.getProperty(key)));
        
        // ===== AUS DATEISYSTEM LADEN (mit UTF-8) =====
        System.out.println("  Messages aus Dateisystem laden (UTF-8):");
        Properties messages = new Properties();
        Path messagesPath = Path.of("src/main/resources/messages_de.properties");
        if (Files.exists(messagesPath)) {
            try (Reader reader = Files.newBufferedReader(messagesPath, StandardCharsets.UTF_8)) {
                messages.load(reader);
                System.out.println("    app.welcome: " + messages.getProperty("app.welcome"));
                System.out.println("    button.save: " + messages.getProperty("button.save"));
                System.out.println("    error.general: " + messages.getProperty("error.general"));
            } catch (IOException e) {
                System.out.println("    Fehler: " + e.getMessage());
            }
        }
        
        // ===== PROPERTIES SPEICHERN =====
        System.out.println("  Properties speichern:");
        Properties userConfig = new Properties();
        userConfig.setProperty("user.name", "Max Mustermann");
        userConfig.setProperty("user.theme", "dark");
        userConfig.setProperty("user.language", "de");
        userConfig.setProperty("last.login", LocalDateTime.now().toString());
        
        Path outputDir = Path.of("output");
        try {
            Files.createDirectories(outputDir);
            Path userConfigPath = outputDir.resolve("user.properties");
            
            try (OutputStream out = Files.newOutputStream(userConfigPath)) {
                userConfig.store(out, "Benutzereinstellungen - Automatisch generiert");
            }
            System.out.println("    Gespeichert: " + userConfigPath);
            
            // Inhalt anzeigen
            System.out.println("    Inhalt:");
            Files.readAllLines(userConfigPath).forEach(line -> 
                System.out.println("      " + line));
                
        } catch (IOException e) {
            System.out.println("    Fehler: " + e.getMessage());
        }
        
        // ===== PROPERTIES ITERIEREN =====
        System.out.println("  Alle Properties iterieren:");
        userConfig.forEach((key, value) -> 
            System.out.println("    " + key + " -> " + value));
    }
}
