package de.javafleet.fileio;

import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.time.LocalDateTime;
import java.util.List;

/**
 * Demonstration: Dateien schreiben.
 */
public class SchreibenDemo {
    
    public static void demo() {
        // Arbeitsverzeichnis für Output
        Path outputDir = Path.of("output");
        
        try {
            // Output-Verzeichnis erstellen falls nicht vorhanden
            Files.createDirectories(outputDir);
            
            // ===== STRING SCHREIBEN =====
            System.out.println("  writeString():");
            Path textDatei = outputDir.resolve("ausgabe.txt");
            Files.writeString(textDatei, "Hallo Welt!\nZeile 2\nZeile 3", StandardCharsets.UTF_8);
            System.out.println("    Geschrieben: " + textDatei);
            
            // ===== LISTE SCHREIBEN =====
            System.out.println("  write() mit Liste:");
            Path listenDatei = outputDir.resolve("liste.txt");
            List<String> zeilen = List.of(
                "Erste Zeile",
                "Zweite Zeile",
                "Dritte Zeile",
                "Erstellt: " + LocalDateTime.now()
            );
            Files.write(listenDatei, zeilen, StandardCharsets.UTF_8);
            System.out.println("    Geschrieben: " + listenDatei);
            
            // ===== ANHÄNGEN STATT ÜBERSCHREIBEN =====
            System.out.println("  Anhängen (APPEND):");
            Path logDatei = outputDir.resolve("log.txt");
            
            // Erste Zeile (erstellt oder überschreibt)
            Files.writeString(logDatei, "=== Log Start ===\n", 
                StandardOpenOption.CREATE, 
                StandardOpenOption.TRUNCATE_EXISTING);
            
            // Weitere Zeilen anhängen
            for (int i = 1; i <= 3; i++) {
                String logEintrag = LocalDateTime.now() + " - Eintrag " + i + "\n";
                Files.writeString(logDatei, logEintrag, StandardOpenOption.APPEND);
            }
            System.out.println("    Log geschrieben: " + logDatei);
            
            // ===== MIT BUFFEREDWRITER =====
            System.out.println("  BufferedWriter:");
            Path bwDatei = outputDir.resolve("buffered.txt");
            try (BufferedWriter writer = Files.newBufferedWriter(bwDatei, StandardCharsets.UTF_8)) {
                writer.write("Zeile 1 mit BufferedWriter");
                writer.newLine();
                writer.write("Zeile 2");
                writer.newLine();
                writer.write("Zeile 3");
            }
            System.out.println("    Geschrieben: " + bwDatei);
            
            // ===== DATEIGRÖSSE ANZEIGEN =====
            System.out.println("  Erstellte Dateien:");
            Files.list(outputDir).forEach(p -> {
                try {
                    System.out.println("    " + p.getFileName() + " (" + Files.size(p) + " bytes)");
                } catch (IOException e) {
                    e.printStackTrace();
                }
            });
            
        } catch (IOException e) {
            System.out.println("    Fehler: " + e.getMessage());
        }
    }
}
