package de.javafleet.fileio;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.stream.Stream;

/**
 * Demonstration: Verzeichnisse erstellen und durchlaufen.
 */
public class VerzeichnisDemo {
    
    public static void demo() {
        // ===== VERZEICHNIS ERSTELLEN =====
        System.out.println("  Verzeichnisse erstellen:");
        Path neuesDir = Path.of("output/demo/unterordner");
        try {
            Files.createDirectories(neuesDir);
            System.out.println("    Erstellt: " + neuesDir);
        } catch (IOException e) {
            System.out.println("    Fehler: " + e.getMessage());
        }
        
        // ===== VERZEICHNIS AUFLISTEN =====
        System.out.println("  Verzeichnis auflisten (list):");
        Path srcDir = Path.of("src/main/java");
        try (Stream<Path> stream = Files.list(srcDir)) {
            stream.forEach(p -> System.out.println("    " + p.getFileName()));
        } catch (IOException e) {
            System.out.println("    Fehler: " + e.getMessage());
        }
        
        // ===== REKURSIV DURCHLAUFEN =====
        System.out.println("  Rekursiv durchlaufen (walk):");
        try (Stream<Path> stream = Files.walk(Path.of("src"), 3)) {
            stream
                .filter(Files::isRegularFile)
                .filter(p -> p.toString().endsWith(".java"))
                .forEach(p -> System.out.println("    " + p));
        } catch (IOException e) {
            System.out.println("    Fehler: " + e.getMessage());
        }
        
        // ===== ALLE DATEIEN EINES TYPS FINDEN =====
        System.out.println("  Alle .properties finden:");
        try (Stream<Path> stream = Files.walk(Path.of("src"))) {
            long count = stream
                .filter(Files::isRegularFile)
                .filter(p -> p.toString().endsWith(".properties"))
                .peek(p -> System.out.println("    " + p))
                .count();
            System.out.println("    Gefunden: " + count);
        } catch (IOException e) {
            System.out.println("    Fehler: " + e.getMessage());
        }
        
        // ===== VERZEICHNISSTRUKTUR ANZEIGEN =====
        System.out.println("  Verzeichnisstruktur (testdaten):");
        Path testdaten = Path.of("testdaten");
        if (Files.exists(testdaten)) {
            try (Stream<Path> stream = Files.list(testdaten)) {
                stream.forEach(p -> {
                    try {
                        String typ = Files.isDirectory(p) ? "[DIR]" : "[FILE]";
                        long size = Files.isRegularFile(p) ? Files.size(p) : 0;
                        System.out.printf("    %s %s (%d bytes)%n", 
                            typ, p.getFileName(), size);
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                });
            } catch (IOException e) {
                System.out.println("    Fehler: " + e.getMessage());
            }
        }
    }
}
