package de.javafleet.annotationsthreads.annotations;

import java.lang.reflect.Field;
import java.lang.reflect.Method;

/**
 * Demonstration: Eigene Annotations erstellen und auswerten.
 */
public class CustomAnnotationsDemo {
    
    public static void demo() throws Exception {
        System.out.println("  Eigene Annotations:");
        
        // === @LogExecutionTime Demo ===
        System.out.println();
        System.out.println("  @LogExecutionTime Demo:");
        BerichtService service = new BerichtService();
        verarbeiteLogAnnotations(service);
        
        // === @NotEmpty und @Range Demo ===
        System.out.println();
        System.out.println("  @NotEmpty und @Range Demo:");
        
        Person validePerson = new Person("Max", 25);
        Person invalidePerson = new Person("", 200);
        
        System.out.println("    Validiere " + validePerson + ":");
        validiere(validePerson);
        
        System.out.println("    Validiere " + invalidePerson + ":");
        validiere(invalidePerson);
    }
    
    /**
     * Verarbeitet @LogExecutionTime Annotations per Reflection.
     */
    private static void verarbeiteLogAnnotations(Object obj) throws Exception {
        Class<?> clazz = obj.getClass();
        
        for (Method method : clazz.getDeclaredMethods()) {
            if (method.isAnnotationPresent(LogExecutionTime.class)) {
                LogExecutionTime annotation = method.getAnnotation(LogExecutionTime.class);
                
                if (annotation.enabled()) {
                    String beschreibung = annotation.value().isEmpty() 
                        ? method.getName() 
                        : annotation.value();
                    
                    System.out.println("    Starte: " + beschreibung);
                    
                    long start = System.currentTimeMillis();
                    method.invoke(obj);  // Methode aufrufen
                    long duration = System.currentTimeMillis() - start;
                    
                    System.out.println("    Fertig: " + beschreibung + " (" + duration + "ms)");
                } else {
                    System.out.println("    Logging deaktiviert für: " + method.getName());
                }
            }
        }
    }
    
    /**
     * Validiert ein Objekt anhand seiner Annotations.
     */
    private static void validiere(Object obj) {
        Class<?> clazz = obj.getClass();
        
        for (Field field : clazz.getDeclaredFields()) {
            field.setAccessible(true);
            
            try {
                // @NotEmpty prüfen
                if (field.isAnnotationPresent(NotEmpty.class)) {
                    NotEmpty notEmpty = field.getAnnotation(NotEmpty.class);
                    Object value = field.get(obj);
                    
                    if (value == null || value.toString().isEmpty()) {
                        System.out.println("      ❌ " + field.getName() + ": " + notEmpty.message());
                    } else {
                        System.out.println("      ✓ " + field.getName() + " ist nicht leer");
                    }
                }
                
                // @Range prüfen
                if (field.isAnnotationPresent(Range.class)) {
                    Range range = field.getAnnotation(Range.class);
                    Object value = field.get(obj);
                    
                    if (value instanceof Integer) {
                        int intValue = (Integer) value;
                        if (intValue < range.min() || intValue > range.max()) {
                            System.out.println("      ❌ " + field.getName() + ": " + range.message() 
                                + " (erwartet: " + range.min() + "-" + range.max() + ")");
                        } else {
                            System.out.println("      ✓ " + field.getName() + " ist im gültigen Bereich");
                        }
                    }
                }
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        }
    }
}

// ===== HILFSKLASSEN =====

class BerichtService {
    
    @LogExecutionTime("Monatsbericht generieren")
    public void generiereMonatsbericht() {
        try {
            Thread.sleep(100);  // Simuliert Arbeit
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }
    
    @LogExecutionTime(value = "Daten exportieren", enabled = false)
    public void exportiereDaten() {
        // Logging ist deaktiviert
    }
    
    @LogExecutionTime  // Verwendet Default
    public void berechneStatistik() {
        try {
            Thread.sleep(50);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }
}

class Person {
    @NotEmpty(message = "Name ist erforderlich")
    private String name;
    
    @Range(min = 0, max = 150, message = "Ungültiges Alter")
    private int alter;
    
    public Person(String name, int alter) {
        this.name = name;
        this.alter = alter;
    }
    
    @Override
    public String toString() {
        return "Person{name='" + name + "', alter=" + alter + "}";
    }
}
