package de.javafleet.annotationsthreads.threads;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.*;

/**
 * Demonstration: Callable und Future für Rückgabewerte.
 */
public class CallableFutureDemo {
    
    public static void demo() throws Exception {
        ExecutorService executor = Executors.newFixedThreadPool(3);
        
        // === EINZELNES CALLABLE ===
        System.out.println("  Callable mit Rückgabewert:");
        
        Callable<Integer> berechnung = () -> {
            Thread.sleep(500);
            return 42;
        };
        
        Future<Integer> future = executor.submit(berechnung);
        
        System.out.println("    Berechnung gestartet...");
        System.out.println("    isDone: " + future.isDone());
        
        // Andere Arbeit während Berechnung läuft
        System.out.println("    Mache andere Arbeit...");
        Thread.sleep(200);
        
        // Ergebnis abholen (blockiert falls noch nicht fertig)
        Integer ergebnis = future.get();  
        System.out.println("    Ergebnis: " + ergebnis);
        System.out.println("    isDone: " + future.isDone());
        
        // === MIT TIMEOUT ===
        System.out.println();
        System.out.println("  Future mit Timeout:");
        
        Future<String> slowFuture = executor.submit(() -> {
            Thread.sleep(100);
            return "Schnell genug!";
        });
        
        try {
            String result = slowFuture.get(2, TimeUnit.SECONDS);
            System.out.println("    Ergebnis: " + result);
        } catch (TimeoutException e) {
            System.out.println("    Timeout! Task abbrechen...");
            slowFuture.cancel(true);
        }
        
        // === MEHRERE FUTURES ===
        System.out.println();
        System.out.println("  Mehrere Callable parallel:");
        
        List<Callable<String>> aufgaben = List.of(
            () -> { Thread.sleep(300); return "Ergebnis A"; },
            () -> { Thread.sleep(100); return "Ergebnis B"; },
            () -> { Thread.sleep(200); return "Ergebnis C"; }
        );
        
        // Alle ausführen und auf alle warten
        List<Future<String>> futures = executor.invokeAll(aufgaben);
        
        System.out.println("    Alle Ergebnisse:");
        for (Future<String> f : futures) {
            System.out.println("      " + f.get());
        }
        
        // === INVOKEANY - Erstes Ergebnis ===
        System.out.println();
        System.out.println("  invokeAny (erstes Ergebnis gewinnt):");
        
        List<Callable<String>> rennen = List.of(
            () -> { Thread.sleep(300); return "Langsam"; },
            () -> { Thread.sleep(50);  return "Schnell"; },
            () -> { Thread.sleep(200); return "Mittel"; }
        );
        
        String gewinner = executor.invokeAny(rennen);
        System.out.println("    Gewinner: " + gewinner);
        
        // Pool beenden
        executor.shutdown();
        executor.awaitTermination(5, TimeUnit.SECONDS);
    }
}
