# Tag 8 - JPA Relationen (OneToOne & ManyToOne)

**Java Web Aufbau - Tag 8 von 10**  
*Demonstrationsprojekt für JPA Entity-Beziehungen*

---

## 📋 Projektübersicht

Dieses Maven-Projekt demonstriert die Verwendung von JPA Entity-Relationen mit Jakarta EE 10:

- **@OneToOne**: User ↔ UserProfile (1:1 Beziehung)
- **@ManyToOne**: Order → User (N:1 Beziehung)
- **CascadeType**: Automatische Propagierung von Operationen
- **FetchType**: LAZY vs. EAGER Loading
- **JOIN FETCH**: Performance-Optimierung gegen N+1 Problem

---

## 🎯 Lernziele

Nach diesem Projekt kannst du:

✅ @OneToOne Relationen mit CascadeType konfigurieren  
✅ @ManyToOne Relationen mit Foreign Keys erstellen  
✅ Bidirektionale Relationen richtig implementieren  
✅ LAZY vs. EAGER Fetch Strategies verstehen  
✅ N+1 Problem mit JOIN FETCH vermeiden  
✅ orphanRemoval für abhängige Entities nutzen  
✅ Production-Ready JPA Code schreiben

---

## 🏗️ Projekt-Struktur

```
tag08-jpa-relationen/
├── pom.xml
├── README.md
└── src/
    └── main/
        ├── java/com/javafleet/
        │   ├── model/
        │   │   ├── User.java              (@OneToOne mit Profile, @OneToMany mit Orders)
        │   │   ├── UserProfile.java       (@OneToOne - abhängig von User)
        │   │   ├── Order.java             (@ManyToOne mit User)
        │   │   └── OrderStatus.java       (Enum für Order Status)
        │   ├── service/
        │   │   └── OrderManagementService.java  (Business Logic mit EntityManager)
        │   └── web/
        │       └── OrderServlet.java      (Demo Servlet)
        ├── resources/
        │   └── META-INF/
        │       └── persistence.xml        (JPA Konfiguration)
        └── webapp/
            ├── index.html                 (Landing Page)
            └── WEB-INF/
                └── beans.xml              (CDI Aktivierung)
```

---

## 📦 Voraussetzungen

### Software

- **JDK 21 LTS** (OpenJDK oder Amazon Corretto)
- **Apache Maven 3.9+**
- **Payara Server 6.x** (oder WildFly 30+)
- **MySQL 8.x** (oder PostgreSQL 15+)
- **Apache NetBeans 22** (optional, aber empfohlen)

### Datenbank Setup

**Option 1: Docker (empfohlen)**

```bash
docker run --name mysql-jpa \
  -e MYSQL_ROOT_PASSWORD=secret \
  -e MYSQL_DATABASE=jpadb \
  -p 3306:3306 -d mysql:8
```

**Option 2: Lokale MySQL Installation**

```sql
CREATE DATABASE jpadb CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
```

---

## 🚀 Installation & Deployment

### 1. Projekt clonen/extrahieren

```bash
unzip tag08-jpa-relationen.zip
cd tag08-jpa-relationen
```

### 2. Maven Build

```bash
mvn clean package
```

Erstellt: `target/tag08-jpa-relationen.war`

### 3. Payara JDBC DataSource konfigurieren

**Admin Console:** http://localhost:4848

1. **JDBC Connection Pool erstellen:**
   - Pool Name: `JPAPool`
   - Resource Type: `javax.sql.DataSource`
   - Database Driver: `com.mysql.cj.jdbc.MysqlDataSource`
   - Properties:
     - `serverName`: `localhost`
     - `portNumber`: `3306`
     - `databaseName`: `jpadb`
     - `user`: `root`
     - `password`: `secret`
     - `useSSL`: `false`
     - `allowPublicKeyRetrieval`: `true`

2. **JDBC Resource erstellen:**
   - JNDI Name: `jdbc/jpadb`
   - Pool Name: `JPAPool`

**Alternativ: CLI**

```bash
asadmin create-jdbc-connection-pool \
  --datasourceclassname com.mysql.cj.jdbc.MysqlDataSource \
  --restype javax.sql.DataSource \
  --property "serverName=localhost:portNumber=3306:databaseName=jpadb:user=root:password=secret:useSSL=false:allowPublicKeyRetrieval=true" \
  JPAPool

asadmin create-jdbc-resource --connectionpoolid JPAPool jdbc/jpadb
```

### 4. WAR deployen

**Option A: Admin Console**
1. Applications → Deploy
2. WAR-Datei auswählen
3. Context Root: `/tag08-jpa-relationen`
4. Deploy klicken

**Option B: CLI**

```bash
asadmin deploy target/tag08-jpa-relationen.war
```

**Option C: NetBeans**
1. Projekt öffnen
2. Rechtsklick → Run

### 5. Anwendung testen

**Landing Page:**  
http://localhost:8080/tag08-jpa-relationen/

**Demo Servlet:**  
http://localhost:8080/tag08-jpa-relationen/orders

---

## 📚 Demo-Funktionen

### 1. Testdaten erstellen

**URL:** `/orders?action=create`

Erstellt:
- 2 Users mit UserProfile (@OneToOne)
- 5 Orders verteilt auf die Users (@ManyToOne)

**Was passiert:**
```java
// @OneToOne mit CascadeType.ALL
User user = new User("alice", "alice@example.com");
UserProfile profile = new UserProfile("Alice", "Johnson");
user.setProfile(profile);
em.persist(user);  // Profile wird automatisch mit persistiert!

// @ManyToOne ohne Cascade
Order order = new Order("ORD-001", new BigDecimal("99.99"), user);
em.persist(order);  // User muss bereits existieren!
```

### 2. Orders auflisten

**URL:** `/orders?action=list`

Zeigt alle Orders mit User-Informationen.

**Technischer Hinweis:**  
Nutzt `JOIN FETCH` um N+1 Problem zu vermeiden:

```java
em.createQuery(
    "SELECT o FROM Order o JOIN FETCH o.user ORDER BY o.orderDate DESC",
    Order.class
).getResultList();
```

### 3. Users anzeigen

**URL:** `/orders?action=users`

Listet alle Users mit:
- Profile-Informationen (@OneToOne)
- Anzahl Orders (@OneToMany)
- Gesamtsumme aller Orders

### 4. User Details

**URL:** `/orders?action=user&id=1`

Zeigt einen User mit:
- Basis-Informationen
- Profile-Details
- Alle zugehörigen Orders

**Technischer Hinweis:**  
Nutzt `LEFT JOIN FETCH` für bidirektionale Relation:

```java
em.createQuery(
    "SELECT DISTINCT u FROM User u LEFT JOIN FETCH u.orders WHERE u.id = :id",
    User.class
).getResultList();
```

---

## 🔧 Entity-Relationen erklärt

### @OneToOne: User ↔ UserProfile

**User.java:**
```java
@OneToOne(cascade = CascadeType.ALL, orphanRemoval = true)
@JoinColumn(name = "profile_id", referencedColumnName = "id")
private UserProfile profile;
```

**Was bedeutet das?**

- **cascade = CascadeType.ALL**: Alle Operationen auf User propagieren zu Profile
  - `persist(user)` → `persist(profile)`
  - `remove(user)` → `remove(profile)`
- **orphanRemoval = true**: Profile ohne User wird gelöscht
  - `user.setProfile(null)` → Profile wird gelöscht
- **@JoinColumn**: Foreign Key `profile_id` in `users` Tabelle

**Datenbank:**
```sql
CREATE TABLE users (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    email VARCHAR(100) NOT NULL,
    profile_id BIGINT,
    FOREIGN KEY (profile_id) REFERENCES user_profiles(id)
);
```

### @ManyToOne: Order → User

**Order.java:**
```java
@ManyToOne(fetch = FetchType.LAZY)
@JoinColumn(name = "user_id", nullable = false)
private User user;
```

**Was bedeutet das?**

- **fetch = FetchType.LAZY**: User wird erst geladen, wenn darauf zugegriffen wird
- **nullable = false**: Jede Order MUSS einem User gehören
- **KEIN Cascade**: Order löschen soll User NICHT löschen!

**Datenbank:**
```sql
CREATE TABLE orders (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    order_number VARCHAR(50) NOT NULL UNIQUE,
    total_amount DECIMAL(19,2) NOT NULL,
    order_date DATETIME,
    status VARCHAR(20) NOT NULL,
    user_id BIGINT NOT NULL,
    FOREIGN KEY (user_id) REFERENCES users(id)
);
```

### Bidirektionale Relation

**User.java (optional):**
```java
@OneToMany(mappedBy = "user", cascade = CascadeType.REMOVE)
private List<Order> orders = new ArrayList<>();
```

**Was bedeutet mappedBy?**

- **"user"** verweist auf `Order.user` Attribut
- User ist die "non-owning" Seite
- Order ist die "owning" Seite (hat @JoinColumn)
- Nur die owning Seite erstellt den Foreign Key!

**Convenience Methods:**
```java
public void addOrder(Order order) {
    orders.add(order);
    order.setUser(this);  // Beide Seiten synchronisieren!
}
```

---

## 🎯 Best Practices

### 1. Immer LAZY für @ManyToOne

```java
@ManyToOne(fetch = FetchType.LAZY)  // ✅ IMMER LAZY!
private User user;
```

**Warum?**
- Bessere Performance
- Nur laden, was benötigt wird
- Verhindert ungewolltes Eager Loading

### 2. JOIN FETCH bei Queries

```java
// ❌ FALSCH: N+1 Problem
List<Order> orders = em.createQuery("SELECT o FROM Order o", Order.class)
    .getResultList();
for (Order o : orders) {
    System.out.println(o.getUser().getUsername());  // N Queries!
}

// ✅ RICHTIG: Eine Query
List<Order> orders = em.createQuery(
    "SELECT o FROM Order o JOIN FETCH o.user",
    Order.class
).getResultList();
```

### 3. CascadeType bewusst einsetzen

```java
// ✅ @OneToOne: CascadeType.ALL okay
@OneToOne(cascade = CascadeType.ALL, orphanRemoval = true)
private UserProfile profile;

// ❌ @ManyToOne: NIEMALS CascadeType.REMOVE!
@ManyToOne(cascade = CascadeType.REMOVE)  // FALSCH!
private User user;
```

**Warum?**  
`remove(order)` würde `user` löschen - und damit alle anderen Orders!

### 4. Bidirektionale Relationen pflegen

```java
// ❌ FALSCH: Nur eine Seite
order.setUser(user);

// ✅ RICHTIG: Beide Seiten synchronisieren
user.addOrder(order);  // Nutzt Convenience Method
```

### 5. orphanRemoval bei Composition

```java
// ✅ Profile gehört zu User (Composition)
@OneToOne(orphanRemoval = true)
private UserProfile profile;

// ❌ Order gehört zu User, aber ist unabhängig (Aggregation)
@OneToMany(orphanRemoval = true)  // NICHT für Orders!
private List<Order> orders;
```

---

## 🐛 Troubleshooting

### Problem: LazyInitializationException

**Fehler:**
```
org.hibernate.LazyInitializationException: could not initialize proxy - no Session
```

**Ursache:** LAZY Entity außerhalb Transaction aufgerufen.

**Lösung 1: JOIN FETCH**
```java
em.createQuery("SELECT o FROM Order o JOIN FETCH o.user WHERE o.id = :id", Order.class)
```

**Lösung 2: @Transactional breiter**
```java
@Transactional
public void processOrder(Long id) {
    Order order = em.find(Order.class, id);
    String username = order.getUser().getUsername();  // OK in Transaction!
}
```

---

### Problem: Foreign Key Constraint Verletzung

**Fehler:**
```
MySQLIntegrityConstraintViolationException: Cannot add or update a child row
```

**Ursache:** User existiert nicht oder user_id ist NULL.

**Lösung:**
```java
User user = em.find(User.class, userId);
if (user == null) {
    throw new IllegalArgumentException("User not found!");
}
Order order = new Order("ORD-001", amount, user);
```

---

### Problem: Infinite Loop bei JSON Serialization

**Fehler:**
```
StackOverflowError during JSON serialization
```

**Ursache:** Bidirektionale Relation ohne `@JsonIgnore`.

**Lösung:**
```java
@ManyToOne
@JsonIgnore  // ✅ Verhindert Loop
private User user;
```

**Oder:** Nutze DTOs statt Entities für JSON!

---

### Problem: DataSource nicht gefunden

**Fehler:**
```
javax.naming.NameNotFoundException: jdbc/jpadb not found
```

**Lösung:**
1. Payara Admin Console → JDBC Resources
2. Stelle sicher: `jdbc/jpadb` existiert
3. Ping Connection Pool testen

---

## 📖 Weiterführende Resources

**Offizielle Docs:**
- Jakarta Persistence: https://jakarta.ee/specifications/persistence/
- Hibernate ORM: https://hibernate.org/orm/documentation/

**Tutorials:**
- Baeldung JPA Relationships: https://www.baeldung.com/jpa-relationships
- Thorben Janssen Associations: https://thorben-janssen.com/ultimate-guide-association-mappings-jpa-hibernate/

**Books:**
- "Pro JPA 2" by Mike Keith
- "High-Performance Java Persistence" by Vlad Mihalcea

---

## 📝 Lizenz & Support

**Projekt:** Java Web Aufbau - Tag 8  
**Autor:** Java Fleet Systems Consulting  
**Website:** https://java-developer.online  
**Support:** support@java-developer.online

---

## 🚀 Nächste Schritte

**Tag 9:** JPA Relationen (2) - @OneToMany & @ManyToMany

Was dich erwartet:
- @OneToMany für 1:N Relationen (User → Orders)
- @ManyToMany für M:N Relationen (Students ↔ Courses)
- Join Tables erstellen und konfigurieren
- Collection-Management in bidirektionalen Relationen

**Viel Erfolg mit JPA Relationen!** 🎉
