package com.javafleet.model;

import jakarta.persistence.*;
import java.util.ArrayList;
import java.util.List;

/**
 * User Entity
 * Tag 8 - JPA Relationen (OneToOne & ManyToOne)
 * 
 * Relationen:
 * - @OneToOne mit UserProfile (bidirektional)
 * - @OneToMany mit Order (bidirektional)
 */
@Entity
@Table(name = "users")
public class User {
    
    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    private Long id;
    
    @Column(nullable = false, unique = true, length = 50)
    private String username;
    
    @Column(nullable = false, length = 100)
    private String email;
    
    /**
     * @OneToOne Relation mit UserProfile
     * - cascade = CascadeType.ALL: Alle Operationen propagieren
     * - orphanRemoval = true: Profile ohne User wird gelöscht
     */
    @OneToOne(cascade = CascadeType.ALL, orphanRemoval = true)
    @JoinColumn(name = "profile_id", referencedColumnName = "id")
    private UserProfile profile;
    
    /**
     * @OneToMany Relation mit Order (bidirektional)
     * - mappedBy = "user": Order.user ist der Owner
     * - cascade = CascadeType.REMOVE: Orders werden gelöscht, wenn User gelöscht wird
     */
    @OneToMany(mappedBy = "user", cascade = CascadeType.REMOVE)
    private List<Order> orders = new ArrayList<>();
    
    // Constructors
    public User() {}
    
    public User(String username, String email) {
        this.username = username;
        this.email = email;
    }
    
    // Convenience Methods für bidirektionale Relation
    public void addOrder(Order order) {
        orders.add(order);
        order.setUser(this);
    }
    
    public void removeOrder(Order order) {
        orders.remove(order);
        order.setUser(null);
    }
    
    // Getters & Setters
    public Long getId() {
        return id;
    }
    
    public void setId(Long id) {
        this.id = id;
    }
    
    public String getUsername() {
        return username;
    }
    
    public void setUsername(String username) {
        this.username = username;
    }
    
    public String getEmail() {
        return email;
    }
    
    public void setEmail(String email) {
        this.email = email;
    }
    
    public UserProfile getProfile() {
        return profile;
    }
    
    public void setProfile(UserProfile profile) {
        this.profile = profile;
    }
    
    public List<Order> getOrders() {
        return orders;
    }
    
    @Override
    public String toString() {
        return "User{" +
                "id=" + id +
                ", username='" + username + '\'' +
                ", email='" + email + '\'' +
                '}';
    }
}
