package com.javafleet.service;

import com.javafleet.model.*;
import jakarta.ejb.Stateless;
import jakarta.persistence.EntityManager;
import jakarta.persistence.PersistenceContext;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.List;

/**
 * OrderManagementService - Business Logic für Order Management
 * Tag 8 - JPA Relationen (OneToOne & ManyToOne)
 * 
 * Demonstriert:
 * - @OneToOne Operationen (User ↔ Profile)
 * - @ManyToOne Operationen (Orders → User)
 * - Cascade Operations
 * - JOIN FETCH für Performance
 */
@Stateless
public class OrderManagementService {
    
    @PersistenceContext
    private EntityManager em;
    
    /**
     * User mit Profile erstellen
     * Dank CascadeType.ALL wird Profile automatisch persistiert
     */
    public User createUserWithProfile(String username, String email,
                                      String firstName, String lastName,
                                      LocalDate dateOfBirth) {
        UserProfile profile = new UserProfile(firstName, lastName);
        profile.setDateOfBirth(dateOfBirth);
        
        User user = new User(username, email);
        user.setProfile(profile);
        
        em.persist(user);  // Profile wird automatisch mit persistiert!
        return user;
    }
    
    /**
     * Order erstellen und User zuordnen
     * Bidirektionale Relation wird mit addOrder() gepflegt
     */
    public Order createOrder(Long userId, String orderNumber, BigDecimal amount) {
        User user = em.find(User.class, userId);
        if (user == null) {
            throw new IllegalArgumentException("User not found: " + userId);
        }
        
        Order order = new Order(orderNumber, amount, user);
        user.addOrder(order);  // Bidirektionale Relation pflegen!
        
        em.persist(order);
        return order;
    }
    
    /**
     * User mit allen Orders laden
     * JOIN FETCH verhindert N+1 Problem
     */
    public User findUserWithOrders(Long userId) {
        List<User> result = em.createQuery(
            "SELECT DISTINCT u FROM User u " +
            "LEFT JOIN FETCH u.orders " +
            "WHERE u.id = :id",
            User.class
        )
        .setParameter("id", userId)
        .getResultList();
        
        return result.isEmpty() ? null : result.get(0);
    }
    
    /**
     * User mit Profile laden
     * Profile wird automatisch geladen (FetchType.EAGER für @OneToOne)
     */
    public User findUserWithProfile(Long userId) {
        return em.find(User.class, userId);
    }
    
    /**
     * Recent Orders mit User-Info laden
     * JOIN FETCH lädt User sofort (verhindert LazyInitializationException)
     */
    public List<Order> findRecentOrders(int limit) {
        return em.createQuery(
            "SELECT o FROM Order o " +
            "JOIN FETCH o.user " +
            "ORDER BY o.orderDate DESC",
            Order.class
        )
        .setMaxResults(limit)
        .getResultList();
    }
    
    /**
     * Orders eines Users finden
     */
    public List<Order> findOrdersByUser(Long userId) {
        return em.createQuery(
            "SELECT o FROM Order o " +
            "WHERE o.user.id = :userId " +
            "ORDER BY o.orderDate DESC",
            Order.class
        )
        .setParameter("userId", userId)
        .getResultList();
    }
    
    /**
     * Order Status aktualisieren
     * Dirty Checking macht automatisches UPDATE
     */
    public void updateOrderStatus(Long orderId, OrderStatus newStatus) {
        Order order = em.find(Order.class, orderId);
        if (order != null) {
            order.setStatus(newStatus);
            // Kein em.merge() nötig! Dirty Checking!
        }
    }
    
    /**
     * User Profile aktualisieren
     * Dirty Checking macht automatisches UPDATE
     */
    public void updateUserProfile(Long userId, String bio, String phone) {
        User user = em.find(User.class, userId);
        if (user != null && user.getProfile() != null) {
            user.getProfile().setBio(bio);
            user.getProfile().setPhoneNumber(phone);
            // Automatisches UPDATE dank Dirty Checking!
        }
    }
    
    /**
     * User mit allen abhängigen Entities löschen
     * - Profile: gelöscht (orphanRemoval = true)
     * - Orders: gelöscht (cascade = CascadeType.REMOVE)
     */
    public void deleteUser(Long userId) {
        User user = em.find(User.class, userId);
        if (user != null) {
            em.remove(user);
            // Profile und Orders werden automatisch gelöscht!
        }
    }
    
    /**
     * Einzelne Order löschen
     * User bleibt erhalten (kein Cascade von Order → User)
     */
    public void deleteOrder(Long orderId) {
        Order order = em.find(Order.class, orderId);
        if (order != null) {
            User user = order.getUser();
            if (user != null) {
                user.removeOrder(order);  // Bidirektionale Relation pflegen!
            }
            em.remove(order);
        }
    }
    
    /**
     * Statistics - Anzahl Orders pro User
     */
    public Long countOrdersByUser(Long userId) {
        return em.createQuery(
            "SELECT COUNT(o) FROM Order o WHERE o.user.id = :userId",
            Long.class
        )
        .setParameter("userId", userId)
        .getSingleResult();
    }
    
    /**
     * Statistics - Gesamtsumme aller Orders eines Users
     */
    public BigDecimal getTotalAmountByUser(Long userId) {
        BigDecimal result = em.createQuery(
            "SELECT SUM(o.totalAmount) FROM Order o WHERE o.user.id = :userId",
            BigDecimal.class
        )
        .setParameter("userId", userId)
        .getSingleResult();
        
        return result != null ? result : BigDecimal.ZERO;
    }
    
    /**
     * Alle Users finden
     */
    public List<User> findAllUsers() {
        return em.createQuery("SELECT u FROM User u ORDER BY u.username", User.class)
            .getResultList();
    }
}
