package com.javafleet.web;

import com.javafleet.model.*;
import com.javafleet.service.OrderManagementService;
import jakarta.inject.Inject;
import jakarta.servlet.ServletException;
import jakarta.servlet.annotation.WebServlet;
import jakarta.servlet.http.HttpServlet;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.List;

/**
 * OrderServlet - Demo Servlet für Order Management
 * Tag 8 - JPA Relationen (OneToOne & ManyToOne)
 * 
 * URLs:
 * - /orders?action=create - Testdaten erstellen
 * - /orders?action=list - Orders auflisten
 * - /orders?action=users - Users auflisten
 * - /orders?action=user&id=1 - User mit Orders anzeigen
 */
@WebServlet("/orders")
public class OrderServlet extends HttpServlet {
    
    @Inject
    private OrderManagementService orderService;
    
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
            throws ServletException, IOException {
        
        response.setContentType("text/html;charset=UTF-8");
        PrintWriter out = response.getWriter();
        
        try {
            out.println("<!DOCTYPE html>");
            out.println("<html>");
            out.println("<head>");
            out.println("<title>Tag 8 - JPA Relationen Demo</title>");
            out.println("<style>");
            out.println("body { font-family: Arial, sans-serif; margin: 20px; }");
            out.println("h1 { color: #2c3e50; }");
            out.println("h2 { color: #34495e; margin-top: 20px; }");
            out.println("table { border-collapse: collapse; width: 100%; margin-top: 10px; }");
            out.println("th, td { border: 1px solid #ddd; padding: 12px; text-align: left; }");
            out.println("th { background-color: #3498db; color: white; }");
            out.println("tr:nth-child(even) { background-color: #f2f2f2; }");
            out.println(".nav { margin: 20px 0; }");
            out.println(".nav a { margin-right: 15px; padding: 10px 15px; background-color: #3498db; color: white; text-decoration: none; border-radius: 4px; }");
            out.println(".nav a:hover { background-color: #2980b9; }");
            out.println(".success { color: #27ae60; font-weight: bold; }");
            out.println(".info { background-color: #ecf0f1; padding: 15px; border-radius: 4px; margin: 10px 0; }");
            out.println("</style>");
            out.println("</head>");
            out.println("<body>");
            
            out.println("<h1>🚀 Tag 8 - JPA Relationen Demo</h1>");
            out.println("<p><strong>Java Web Aufbau - @OneToOne & @ManyToOne</strong></p>");
            
            out.println("<div class='nav'>");
            out.println("<a href='?action=create'>Testdaten erstellen</a>");
            out.println("<a href='?action=users'>Users anzeigen</a>");
            out.println("<a href='?action=list'>Orders anzeigen</a>");
            out.println("<a href='?'>Home</a>");
            out.println("</div>");
            
            String action = request.getParameter("action");
            
            if ("create".equals(action)) {
                handleCreate(out);
            } else if ("list".equals(action)) {
                handleList(out);
            } else if ("users".equals(action)) {
                handleUsers(out);
            } else if ("user".equals(action)) {
                handleUserDetails(request, out);
            } else {
                handleHome(out);
            }
            
            out.println("</body>");
            out.println("</html>");
            
        } catch (Exception e) {
            out.println("<div style='color: red;'>");
            out.println("<h2>Fehler aufgetreten:</h2>");
            out.println("<pre>" + e.getMessage() + "</pre>");
            e.printStackTrace();
            out.println("</div>");
        }
    }
    
    private void handleHome(PrintWriter out) {
        out.println("<h2>Willkommen zur JPA Relationen Demo!</h2>");
        out.println("<div class='info'>");
        out.println("<h3>Was wird demonstriert?</h3>");
        out.println("<ul>");
        out.println("<li><strong>@OneToOne:</strong> User ↔ UserProfile (ein User hat EIN Profile)</li>");
        out.println("<li><strong>@ManyToOne:</strong> Order → User (viele Orders gehören zu einem User)</li>");
        out.println("<li><strong>CascadeType.ALL:</strong> Profile wird mit User gelöscht</li>");
        out.println("<li><strong>orphanRemoval:</strong> Profile ohne User wird gelöscht</li>");
        out.println("<li><strong>FetchType.LAZY:</strong> User wird erst geladen, wenn benötigt</li>");
        out.println("<li><strong>JOIN FETCH:</strong> Vermeidet N+1 Problem</li>");
        out.println("</ul>");
        out.println("</div>");
        
        out.println("<h3>📋 Schritte:</h3>");
        out.println("<ol>");
        out.println("<li>Klicke auf <strong>Testdaten erstellen</strong> um User und Orders anzulegen</li>");
        out.println("<li>Klicke auf <strong>Users anzeigen</strong> um alle Users zu sehen</li>");
        out.println("<li>Klicke auf <strong>Orders anzeigen</strong> um alle Orders zu sehen</li>");
        out.println("</ol>");
    }
    
    private void handleCreate(PrintWriter out) {
        out.println("<h2>✅ Testdaten erstellen</h2>");
        
        // User 1 mit Profile erstellen
        User user1 = orderService.createUserWithProfile(
            "alice", "alice@example.com",
            "Alice", "Johnson",
            LocalDate.of(1990, 5, 15)
        );
        
        out.println("<p class='success'>✓ User erstellt: " + user1.getUsername() + "</p>");
        out.println("<div class='info'>");
        out.println("<strong>ID:</strong> " + user1.getId() + "<br>");
        out.println("<strong>Username:</strong> " + user1.getUsername() + "<br>");
        out.println("<strong>Email:</strong> " + user1.getEmail() + "<br>");
        out.println("<strong>Profile:</strong> " + user1.getProfile().getFirstName() + 
                   " " + user1.getProfile().getLastName() + "<br>");
        out.println("<strong>Geburtsdatum:</strong> " + user1.getProfile().getDateOfBirth());
        out.println("</div>");
        
        // User 2 mit Profile erstellen
        User user2 = orderService.createUserWithProfile(
            "bob", "bob@example.com",
            "Bob", "Smith",
            LocalDate.of(1985, 8, 22)
        );
        
        out.println("<p class='success'>✓ User erstellt: " + user2.getUsername() + "</p>");
        
        // Orders für User 1 erstellen
        Order order1 = orderService.createOrder(
            user1.getId(), "ORD-001", new BigDecimal("99.99")
        );
        Order order2 = orderService.createOrder(
            user1.getId(), "ORD-002", new BigDecimal("149.50")
        );
        Order order3 = orderService.createOrder(
            user1.getId(), "ORD-003", new BigDecimal("79.00")
        );
        
        out.println("<h3>Orders für " + user1.getUsername() + ":</h3>");
        out.println("<ul>");
        out.println("<li>" + order1.getOrderNumber() + " - €" + order1.getTotalAmount() + " - " + order1.getStatus() + "</li>");
        out.println("<li>" + order2.getOrderNumber() + " - €" + order2.getTotalAmount() + " - " + order2.getStatus() + "</li>");
        out.println("<li>" + order3.getOrderNumber() + " - €" + order3.getTotalAmount() + " - " + order3.getStatus() + "</li>");
        out.println("</ul>");
        
        // Orders für User 2 erstellen
        Order order4 = orderService.createOrder(
            user2.getId(), "ORD-004", new BigDecimal("199.99")
        );
        Order order5 = orderService.createOrder(
            user2.getId(), "ORD-005", new BigDecimal("49.95")
        );
        
        out.println("<h3>Orders für " + user2.getUsername() + ":</h3>");
        out.println("<ul>");
        out.println("<li>" + order4.getOrderNumber() + " - €" + order4.getTotalAmount() + " - " + order4.getStatus() + "</li>");
        out.println("<li>" + order5.getOrderNumber() + " - €" + order5.getTotalAmount() + " - " + order5.getStatus() + "</li>");
        out.println("</ul>");
        
        out.println("<p class='success'>✓ Testdaten erfolgreich erstellt!</p>");
    }
    
    private void handleList(PrintWriter out) {
        out.println("<h2>📦 Alle Orders</h2>");
        
        List<Order> orders = orderService.findRecentOrders(20);
        
        if (orders.isEmpty()) {
            out.println("<p>Keine Orders vorhanden. <a href='?action=create'>Testdaten erstellen</a></p>");
            return;
        }
        
        out.println("<p><strong>Anzahl Orders:</strong> " + orders.size() + "</p>");
        out.println("<table>");
        out.println("<tr>");
        out.println("<th>Order#</th>");
        out.println("<th>User</th>");
        out.println("<th>Email</th>");
        out.println("<th>Betrag</th>");
        out.println("<th>Status</th>");
        out.println("<th>Datum</th>");
        out.println("</tr>");
        
        for (Order order : orders) {
            out.println("<tr>");
            out.println("<td>" + order.getOrderNumber() + "</td>");
            out.println("<td>" + order.getUser().getUsername() + "</td>");
            out.println("<td>" + order.getUser().getEmail() + "</td>");
            out.println("<td>€" + order.getTotalAmount() + "</td>");
            out.println("<td>" + order.getStatus() + "</td>");
            out.println("<td>" + order.getOrderDate() + "</td>");
            out.println("</tr>");
        }
        
        out.println("</table>");
        
        out.println("<div class='info'>");
        out.println("<strong>💡 Technischer Hinweis:</strong><br>");
        out.println("Diese Query nutzt <code>JOIN FETCH</code> um User-Daten sofort zu laden:<br>");
        out.println("<code>SELECT o FROM Order o JOIN FETCH o.user</code><br>");
        out.println("→ Verhindert LazyInitializationException und N+1 Problem!");
        out.println("</div>");
    }
    
    private void handleUsers(PrintWriter out) {
        out.println("<h2>👥 Alle Users</h2>");
        
        List<User> users = orderService.findAllUsers();
        
        if (users.isEmpty()) {
            out.println("<p>Keine Users vorhanden. <a href='?action=create'>Testdaten erstellen</a></p>");
            return;
        }
        
        out.println("<table>");
        out.println("<tr>");
        out.println("<th>ID</th>");
        out.println("<th>Username</th>");
        out.println("<th>Email</th>");
        out.println("<th>Name</th>");
        out.println("<th>Anzahl Orders</th>");
        out.println("<th>Gesamt</th>");
        out.println("<th>Details</th>");
        out.println("</tr>");
        
        for (User user : users) {
            Long orderCount = orderService.countOrdersByUser(user.getId());
            BigDecimal total = orderService.getTotalAmountByUser(user.getId());
            
            out.println("<tr>");
            out.println("<td>" + user.getId() + "</td>");
            out.println("<td>" + user.getUsername() + "</td>");
            out.println("<td>" + user.getEmail() + "</td>");
            
            if (user.getProfile() != null) {
                out.println("<td>" + user.getProfile().getFirstName() + 
                           " " + user.getProfile().getLastName() + "</td>");
            } else {
                out.println("<td>-</td>");
            }
            
            out.println("<td>" + orderCount + "</td>");
            out.println("<td>€" + total + "</td>");
            out.println("<td><a href='?action=user&id=" + user.getId() + "'>Details</a></td>");
            out.println("</tr>");
        }
        
        out.println("</table>");
    }
    
    private void handleUserDetails(HttpServletRequest request, PrintWriter out) {
        String idParam = request.getParameter("id");
        if (idParam == null) {
            out.println("<p>Keine User-ID angegeben!</p>");
            return;
        }
        
        Long userId = Long.parseLong(idParam);
        User user = orderService.findUserWithOrders(userId);
        
        if (user == null) {
            out.println("<p>User nicht gefunden!</p>");
            return;
        }
        
        out.println("<h2>👤 User Details: " + user.getUsername() + "</h2>");
        
        out.println("<div class='info'>");
        out.println("<h3>Basis-Informationen:</h3>");
        out.println("<strong>ID:</strong> " + user.getId() + "<br>");
        out.println("<strong>Username:</strong> " + user.getUsername() + "<br>");
        out.println("<strong>Email:</strong> " + user.getEmail() + "<br>");
        
        if (user.getProfile() != null) {
            out.println("<h3>Profile (@OneToOne):</h3>");
            out.println("<strong>Name:</strong> " + user.getProfile().getFirstName() + 
                       " " + user.getProfile().getLastName() + "<br>");
            out.println("<strong>Geburtsdatum:</strong> " + user.getProfile().getDateOfBirth() + "<br>");
            if (user.getProfile().getBio() != null) {
                out.println("<strong>Bio:</strong> " + user.getProfile().getBio() + "<br>");
            }
            if (user.getProfile().getPhoneNumber() != null) {
                out.println("<strong>Telefon:</strong> " + user.getProfile().getPhoneNumber() + "<br>");
            }
        }
        out.println("</div>");
        
        out.println("<h3>📦 Orders (@ManyToOne):</h3>");
        
        if (user.getOrders().isEmpty()) {
            out.println("<p>Keine Orders vorhanden.</p>");
        } else {
            out.println("<table>");
            out.println("<tr>");
            out.println("<th>Order#</th>");
            out.println("<th>Betrag</th>");
            out.println("<th>Status</th>");
            out.println("<th>Datum</th>");
            out.println("</tr>");
            
            for (Order order : user.getOrders()) {
                out.println("<tr>");
                out.println("<td>" + order.getOrderNumber() + "</td>");
                out.println("<td>€" + order.getTotalAmount() + "</td>");
                out.println("<td>" + order.getStatus() + "</td>");
                out.println("<td>" + order.getOrderDate() + "</td>");
                out.println("</tr>");
            }
            
            out.println("</table>");
            
            BigDecimal total = orderService.getTotalAmountByUser(userId);
            out.println("<p><strong>Gesamtsumme:</strong> €" + total + "</p>");
        }
        
        out.println("<div class='info'>");
        out.println("<strong>💡 Technischer Hinweis:</strong><br>");
        out.println("User und Orders wurden mit <code>LEFT JOIN FETCH</code> geladen:<br>");
        out.println("<code>SELECT DISTINCT u FROM User u LEFT JOIN FETCH u.orders WHERE u.id = :id</code><br>");
        out.println("→ Nur EINE Query statt 1+N Queries!");
        out.println("</div>");
    }
}
