package de.javafleet.sync;

import java.util.*;
import java.util.concurrent.*;

/**
 * Demonstration: Thread-sichere Collections.
 */
public class ConcurrentCollectionsDemo {
    
    public static void demo() throws InterruptedException {
        // === CONCURRENTHASHMAP ===
        System.out.println("  ConcurrentHashMap:");
        
        ConcurrentHashMap<String, Integer> map = new ConcurrentHashMap<>();
        
        ExecutorService executor = Executors.newFixedThreadPool(5);
        
        for (int i = 0; i < 100; i++) {
            final int value = i;
            executor.execute(() -> {
                map.put("key" + value, value);
            });
        }
        
        executor.shutdown();
        executor.awaitTermination(5, TimeUnit.SECONDS);
        
        System.out.println("    Anzahl Einträge: " + map.size());
        
        // Atomare Operationen
        map.putIfAbsent("newKey", 999);
        map.computeIfAbsent("computed", k -> 42);
        
        System.out.println("    putIfAbsent: " + map.get("newKey"));
        System.out.println("    computeIfAbsent: " + map.get("computed"));
        
        // === COPYONWRITEARRAYLIST ===
        System.out.println();
        System.out.println("  CopyOnWriteArrayList:");
        
        CopyOnWriteArrayList<String> cowList = new CopyOnWriteArrayList<>();
        cowList.add("A");
        cowList.add("B");
        cowList.add("C");
        
        // Safe iteration - keine ConcurrentModificationException
        for (String item : cowList) {
            System.out.println("    Item: " + item);
            // Kann während Iteration hinzufügen (wird in Kopie gemacht)
            if (item.equals("B")) {
                cowList.add("D");
            }
        }
        System.out.println("    Liste nach Iteration: " + cowList);
        
        // === BLOCKINGQUEUE ===
        System.out.println();
        System.out.println("  BlockingQueue (Producer-Consumer):");
        
        BlockingQueue<String> queue = new LinkedBlockingQueue<>(5);
        
        // Producer
        Thread producer = new Thread(() -> {
            try {
                for (int i = 1; i <= 5; i++) {
                    String item = "Item-" + i;
                    queue.put(item);  // Blockiert wenn voll
                    System.out.println("    Produziert: " + item);
                    Thread.sleep(100);
                }
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        });
        
        // Consumer
        Thread consumer = new Thread(() -> {
            try {
                for (int i = 1; i <= 5; i++) {
                    String item = queue.take();  // Blockiert wenn leer
                    System.out.println("    Konsumiert: " + item);
                    Thread.sleep(200);
                }
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        });
        
        producer.start();
        consumer.start();
        
        producer.join();
        consumer.join();
        
        System.out.println("    Producer-Consumer beendet!");
    }
}
