package de.javafleet.sync;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

/**
 * Demonstration: Race Condition Problem.
 */
public class RaceConditionDemo {
    
    public static void demo() throws InterruptedException {
        System.out.println("  Race Condition mit unsicherem Konto:");
        
        // Unsicheres Konto
        UnsicheresKonto konto = new UnsicheresKonto(1000);
        
        ExecutorService executor = Executors.newFixedThreadPool(10);
        
        // 100 Threads heben jeweils 10€ ab
        for (int i = 0; i < 100; i++) {
            executor.execute(() -> {
                konto.abheben(10);
            });
        }
        
        executor.shutdown();
        executor.awaitTermination(5, TimeUnit.SECONDS);
        
        // Erwarteter Kontostand: 1000 - (100 * 10) = 0
        System.out.println("    Erwarteter Kontostand: 0");
        System.out.println("    Tatsächlicher Kontostand: " + konto.getKontostand());
        System.out.println("    → Race Condition verursacht inkonsistente Ergebnisse!");
    }
}

/**
 * NICHT thread-safe! Nur zur Demonstration.
 */
class UnsicheresKonto {
    private int kontostand;
    
    public UnsicheresKonto(int kontostand) {
        this.kontostand = kontostand;
    }
    
    public void abheben(int betrag) {
        if (kontostand >= betrag) {
            // Hier kann ein Thread-Wechsel passieren!
            try { Thread.sleep(1); } catch (InterruptedException e) {}
            kontostand -= betrag;
        }
    }
    
    public int getKontostand() {
        return kontostand;
    }
}
