package de.javafleet.sync;

import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.*;
import java.util.concurrent.locks.*;

/**
 * ✅ LÖSUNG: Synchronisation Challenges
 */
public class SyncChallenge {
    
    public static void runChallenges() throws Exception {
        System.out.println("=== Synchronisation Challenges - LÖSUNGEN ===");
        System.out.println();
        
        level1();
        level2();
        level3();
    }
    
    static void level1() throws InterruptedException {
        System.out.println("--- Level 1: Grundlagen ---");
        
        // 1.1: Synchronized Counter
        SimpleCounter counter = new SimpleCounter();
        
        ExecutorService executor = Executors.newFixedThreadPool(10);
        for (int i = 0; i < 1000; i++) {
            executor.execute(() -> counter.increment());
        }
        executor.shutdown();
        executor.awaitTermination(5, TimeUnit.SECONDS);
        
        System.out.println("  1.1 SynchronizedCounter: " + counter.getCount());
        
        // 1.2: AtomicInteger Counter
        AtomicCounter atomicCounter = new AtomicCounter();
        
        ExecutorService executor2 = Executors.newFixedThreadPool(10);
        for (int i = 0; i < 1000; i++) {
            executor2.execute(() -> atomicCounter.increment());
        }
        executor2.shutdown();
        executor2.awaitTermination(5, TimeUnit.SECONDS);
        
        System.out.println("  1.2 AtomicCounter: " + atomicCounter.getCount());
        
        System.out.println();
    }
    
    static void level2() throws InterruptedException {
        System.out.println("--- Level 2: Locks ---");
        
        // 2.1: ThreadSafeStack
        ThreadSafeStack<Integer> stack = new ThreadSafeStack<>();
        
        ExecutorService executor = Executors.newFixedThreadPool(5);
        for (int i = 0; i < 10; i++) {
            final int value = i;
            executor.execute(() -> stack.push(value));
        }
        executor.shutdown();
        executor.awaitTermination(5, TimeUnit.SECONDS);
        
        System.out.println("  2.1 Stack size: " + stack.size());
        System.out.println("      Pop: " + stack.pop());
        
        // 2.2: Cache mit ReadWriteLock
        ThreadSafeCache cache = new ThreadSafeCache();
        
        ExecutorService executor2 = Executors.newFixedThreadPool(10);
        
        // Writer
        for (int i = 0; i < 5; i++) {
            final int key = i;
            executor2.execute(() -> cache.put("key" + key, "value" + key));
        }
        
        // Reader
        for (int i = 0; i < 10; i++) {
            final int key = i % 5;
            executor2.execute(() -> cache.get("key" + key));
        }
        
        executor2.shutdown();
        executor2.awaitTermination(5, TimeUnit.SECONDS);
        
        System.out.println("  2.2 Cache size: " + cache.size());
        
        System.out.println();
    }
    
    static void level3() throws Exception {
        System.out.println("--- Level 3: Fortgeschritten ---");
        
        // 3.1: ConnectionPool mit Semaphore
        ConnectionPool pool = new ConnectionPool(3);
        
        ExecutorService executor = Executors.newFixedThreadPool(10);
        for (int i = 0; i < 10; i++) {
            final int id = i;
            executor.execute(() -> {
                try {
                    pool.useConnection(id);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            });
        }
        executor.shutdown();
        executor.awaitTermination(10, TimeUnit.SECONDS);
        
        System.out.println("  3.1 ConnectionPool beendet");
        
        // 3.2: CountDownLatch
        System.out.println("  3.2 CountDownLatch Demo:");
        
        int workers = 3;
        CountDownLatch latch = new CountDownLatch(workers);
        
        for (int i = 0; i < workers; i++) {
            final int id = i;
            new Thread(() -> {
                System.out.println("      Worker " + id + " arbeitet...");
                try { Thread.sleep(500); } catch (InterruptedException e) {}
                System.out.println("      Worker " + id + " fertig!");
                latch.countDown();
            }).start();
        }
        
        latch.await();
        System.out.println("      Alle Worker fertig!");
        
        System.out.println();
    }
}

// ===== LÖSUNGEN =====

class SimpleCounter {
    private int count = 0;
    
    public synchronized void increment() {
        count++;
    }
    
    public synchronized int getCount() {
        return count;
    }
}

class AtomicCounter {
    private AtomicInteger count = new AtomicInteger(0);
    
    public void increment() {
        count.incrementAndGet();
    }
    
    public int getCount() {
        return count.get();
    }
}

class ThreadSafeStack<T> {
    private final Deque<T> stack = new ArrayDeque<>();
    private final ReentrantLock lock = new ReentrantLock();
    
    public void push(T item) {
        lock.lock();
        try {
            stack.push(item);
        } finally {
            lock.unlock();
        }
    }
    
    public T pop() {
        lock.lock();
        try {
            return stack.isEmpty() ? null : stack.pop();
        } finally {
            lock.unlock();
        }
    }
    
    public int size() {
        lock.lock();
        try {
            return stack.size();
        } finally {
            lock.unlock();
        }
    }
}

class ThreadSafeCache {
    private final Map<String, String> data = new HashMap<>();
    private final ReadWriteLock rwLock = new ReentrantReadWriteLock();
    
    public String get(String key) {
        rwLock.readLock().lock();
        try {
            return data.get(key);
        } finally {
            rwLock.readLock().unlock();
        }
    }
    
    public void put(String key, String value) {
        rwLock.writeLock().lock();
        try {
            data.put(key, value);
        } finally {
            rwLock.writeLock().unlock();
        }
    }
    
    public int size() {
        rwLock.readLock().lock();
        try {
            return data.size();
        } finally {
            rwLock.readLock().unlock();
        }
    }
}

class ConnectionPool {
    private final Semaphore semaphore;
    
    public ConnectionPool(int maxConnections) {
        this.semaphore = new Semaphore(maxConnections);
    }
    
    public void useConnection(int id) throws InterruptedException {
        semaphore.acquire();
        try {
            System.out.println("      Verbindung " + id + " aktiv (verfügbar: " + semaphore.availablePermits() + ")");
            Thread.sleep(500);
        } finally {
            semaphore.release();
        }
    }
}
