package de.javafleet.sync;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

/**
 * Demonstration: synchronized Keyword.
 */
public class SynchronizedDemo {
    
    public static void demo() throws InterruptedException {
        System.out.println("  Synchronized Konto (thread-safe):");
        
        SicheresKonto konto = new SicheresKonto(1000);
        
        ExecutorService executor = Executors.newFixedThreadPool(10);
        
        // 100 Threads heben jeweils 10€ ab
        for (int i = 0; i < 100; i++) {
            executor.execute(() -> {
                konto.abheben(10);
            });
        }
        
        executor.shutdown();
        executor.awaitTermination(5, TimeUnit.SECONDS);
        
        System.out.println("    Erwarteter Kontostand: 0");
        System.out.println("    Tatsächlicher Kontostand: " + konto.getKontostand());
        System.out.println("    → Mit synchronized: Immer korrekt!");
        
        // === SYNCHRONIZED BLOCK ===
        System.out.println();
        System.out.println("  Synchronized Block Demo:");
        
        KontoMitBlock kontoBlock = new KontoMitBlock(500);
        
        ExecutorService executor2 = Executors.newFixedThreadPool(5);
        for (int i = 0; i < 50; i++) {
            executor2.execute(() -> kontoBlock.abheben(10));
        }
        
        executor2.shutdown();
        executor2.awaitTermination(5, TimeUnit.SECONDS);
        
        System.out.println("    Kontostand mit Block: " + kontoBlock.getKontostand());
    }
}

/**
 * Thread-safe mit synchronized Methoden.
 */
class SicheresKonto {
    private int kontostand;
    
    public SicheresKonto(int kontostand) {
        this.kontostand = kontostand;
    }
    
    public synchronized void abheben(int betrag) {
        if (kontostand >= betrag) {
            try { Thread.sleep(1); } catch (InterruptedException e) {}
            kontostand -= betrag;
        }
    }
    
    public synchronized void einzahlen(int betrag) {
        kontostand += betrag;
    }
    
    public synchronized int getKontostand() {
        return kontostand;
    }
}

/**
 * Thread-safe mit synchronized Block.
 */
class KontoMitBlock {
    private int kontostand;
    private final Object lock = new Object();
    
    public KontoMitBlock(int kontostand) {
        this.kontostand = kontostand;
    }
    
    public void abheben(int betrag) {
        // Nicht-kritischer Code kann parallel laufen
        System.out.print(""); // Simuliert andere Arbeit
        
        synchronized (lock) {
            // Nur dieser Block ist geschützt
            if (kontostand >= betrag) {
                kontostand -= betrag;
            }
        }
    }
    
    public int getKontostand() {
        synchronized (lock) {
            return kontostand;
        }
    }
}
