package de.javafleet.sync;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.*;
import java.util.concurrent.locks.*;

/**
 * Demonstration: ReentrantLock und ReadWriteLock.
 */
public class LockDemo {
    
    public static void demo() throws InterruptedException {
        // === REENTRANT LOCK ===
        System.out.println("  ReentrantLock Demo:");
        
        KontoMitLock konto = new KontoMitLock(1000);
        
        ExecutorService executor = Executors.newFixedThreadPool(5);
        for (int i = 0; i < 50; i++) {
            executor.execute(() -> konto.abheben(20));
        }
        
        executor.shutdown();
        executor.awaitTermination(5, TimeUnit.SECONDS);
        
        System.out.println("    Kontostand: " + konto.getKontostand());
        
        // === TRYLOCK MIT TIMEOUT ===
        System.out.println();
        System.out.println("  tryLock mit Timeout:");
        
        boolean success = konto.abhebenMitTimeout(100);
        System.out.println("    Abhebung erfolgreich: " + success);
        
        // === READWRITE LOCK ===
        System.out.println();
        System.out.println("  ReadWriteLock Demo:");
        
        Cache cache = new Cache();
        
        ExecutorService executor2 = Executors.newFixedThreadPool(10);
        
        // 5 Writer
        for (int i = 0; i < 5; i++) {
            final int key = i;
            executor2.execute(() -> {
                cache.put("key" + key, "value" + key);
                System.out.println("    Geschrieben: key" + key);
            });
        }
        
        // 10 Reader
        for (int i = 0; i < 10; i++) {
            final int key = i % 5;
            executor2.execute(() -> {
                String value = cache.get("key" + key);
                System.out.println("    Gelesen: key" + key + " = " + value);
            });
        }
        
        executor2.shutdown();
        executor2.awaitTermination(5, TimeUnit.SECONDS);
    }
}

/**
 * Konto mit ReentrantLock.
 */
class KontoMitLock {
    private int kontostand;
    private final ReentrantLock lock = new ReentrantLock();
    
    public KontoMitLock(int kontostand) {
        this.kontostand = kontostand;
    }
    
    public void abheben(int betrag) {
        lock.lock();
        try {
            if (kontostand >= betrag) {
                kontostand -= betrag;
            }
        } finally {
            lock.unlock();  // IMMER in finally!
        }
    }
    
    public boolean abhebenMitTimeout(int betrag) {
        try {
            if (lock.tryLock(1, TimeUnit.SECONDS)) {
                try {
                    if (kontostand >= betrag) {
                        kontostand -= betrag;
                        return true;
                    }
                } finally {
                    lock.unlock();
                }
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        return false;
    }
    
    public int getKontostand() {
        lock.lock();
        try {
            return kontostand;
        } finally {
            lock.unlock();
        }
    }
}

/**
 * Cache mit ReadWriteLock.
 */
class Cache {
    private final Map<String, String> data = new HashMap<>();
    private final ReadWriteLock rwLock = new ReentrantReadWriteLock();
    private final Lock readLock = rwLock.readLock();
    private final Lock writeLock = rwLock.writeLock();
    
    public String get(String key) {
        readLock.lock();  // Mehrere Leser parallel OK
        try {
            return data.get(key);
        } finally {
            readLock.unlock();
        }
    }
    
    public void put(String key, String value) {
        writeLock.lock();  // Exklusiv
        try {
            data.put(key, value);
        } finally {
            writeLock.unlock();
        }
    }
}
