package de.javafleet.network;

import java.net.URI;
import java.net.http.*;
import java.util.List;
import java.util.concurrent.CompletableFuture;

/**
 * Demonstration: Asynchrone HTTP-Requests.
 */
public class AsyncHttpDemo {
    
    public static void demo() throws Exception {
        HttpClient client = HttpClient.newHttpClient();
        
        // ===== EINZELNER ASYNC REQUEST =====
        System.out.println("  Async Request:");
        
        HttpRequest request = HttpRequest.newBuilder()
            .uri(URI.create("https://jsonplaceholder.typicode.com/posts/1"))
            .build();
        
        CompletableFuture<HttpResponse<String>> future = 
            client.sendAsync(request, HttpResponse.BodyHandlers.ofString());
        
        System.out.println("    Request gesendet (nicht blockierend)...");
        
        // Callback wenn fertig
        future.thenAccept(response -> {
            System.out.println("    Async Antwort erhalten!");
            System.out.println("    Status: " + response.statusCode());
        });
        
        // Warten (nur für Demo)
        future.join();
        
        // ===== MEHRERE PARALLELE REQUESTS =====
        System.out.println();
        System.out.println("  Parallele Requests:");
        
        List<String> urls = List.of(
            "https://jsonplaceholder.typicode.com/posts/1",
            "https://jsonplaceholder.typicode.com/posts/2",
            "https://jsonplaceholder.typicode.com/posts/3"
        );
        
        long startTime = System.currentTimeMillis();
        
        List<CompletableFuture<String>> futures = urls.stream()
            .map(url -> HttpRequest.newBuilder().uri(URI.create(url)).build())
            .map(req -> client.sendAsync(req, HttpResponse.BodyHandlers.ofString())
                .thenApply(HttpResponse::body)
                .thenApply(body -> body.substring(0, Math.min(50, body.length()))))
            .toList();
        
        // Auf alle warten
        CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).join();
        
        long duration = System.currentTimeMillis() - startTime;
        
        System.out.println("    " + urls.size() + " Requests parallel in " + duration + "ms");
        
        for (int i = 0; i < futures.size(); i++) {
            System.out.println("    Response " + (i+1) + ": " + futures.get(i).join() + "...");
        }
    }
}
