package de.javafleet.network;

import java.net.URI;
import java.net.http.*;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * ✅ LÖSUNG: Netzwerk Challenges
 */
public class NetworkChallenge {
    
    private static final HttpClient client = HttpClient.newBuilder()
        .connectTimeout(Duration.ofSeconds(10))
        .build();
    
    public static void runChallenges() throws Exception {
        System.out.println("=== Netzwerk Challenges - LÖSUNGEN ===");
        System.out.println();
        
        level1();
        level2();
        level3();
    }
    
    static void level1() throws Exception {
        System.out.println("--- Level 1: Grundlagen ---");
        
        // 1.1: GitHub API aufrufen
        System.out.println("  1.1 GitHub API aufrufen:");
        
        HttpRequest request = HttpRequest.newBuilder()
            .uri(URI.create("https://api.github.com/users/octocat"))
            .header("Accept", "application/json")
            .GET()
            .build();
        
        HttpResponse<String> response = client.send(request, 
            HttpResponse.BodyHandlers.ofString());
        
        System.out.println("    Status: " + response.statusCode());
        
        // 1.2: Name extrahieren
        System.out.println("  1.2 Name extrahieren:");
        
        String body = response.body();
        Pattern namePattern = Pattern.compile("\"name\"\\s*:\\s*\"([^\"]+)\"");
        Matcher matcher = namePattern.matcher(body);
        if (matcher.find()) {
            System.out.println("    Name: " + matcher.group(1));
        }
        
        // 1.3: Headers anzeigen
        System.out.println("  1.3 Response Headers:");
        
        HttpRequest httpbinRequest = HttpRequest.newBuilder()
            .uri(URI.create("https://httpbin.org/get"))
            .build();
        
        HttpResponse<String> httpbinResponse = client.send(httpbinRequest, 
            HttpResponse.BodyHandlers.ofString());
        
        httpbinResponse.headers().map().entrySet().stream()
            .limit(5)
            .forEach(e -> System.out.println("    " + e.getKey() + ": " + e.getValue().get(0)));
        
        System.out.println();
    }
    
    static void level2() throws Exception {
        System.out.println("--- Level 2: POST Requests ---");
        
        // 2.1 & 2.2: POST Request mit JSON
        System.out.println("  2.1/2.2 POST Request:");
        
        String jsonBody = """
            {
                "title": "Mein Titel",
                "body": "Mein Text",
                "userId": 1
            }
            """;
        
        HttpRequest postRequest = HttpRequest.newBuilder()
            .uri(URI.create("https://jsonplaceholder.typicode.com/posts"))
            .header("Content-Type", "application/json")
            .POST(HttpRequest.BodyPublishers.ofString(jsonBody))
            .build();
        
        HttpResponse<String> postResponse = client.send(postRequest, 
            HttpResponse.BodyHandlers.ofString());
        
        System.out.println("    Status: " + postResponse.statusCode());
        System.out.println("    Response: " + postResponse.body().substring(0, 
            Math.min(100, postResponse.body().length())) + "...");
        
        System.out.println();
    }
    
    static void level3() throws Exception {
        System.out.println("--- Level 3: Async & Parallel ---");
        
        // 3.1: Parallele Requests
        System.out.println("  3.1 Parallele Requests:");
        
        List<String> urls = List.of(
            "https://jsonplaceholder.typicode.com/posts/1",
            "https://jsonplaceholder.typicode.com/posts/2",
            "https://jsonplaceholder.typicode.com/posts/3",
            "https://jsonplaceholder.typicode.com/posts/4",
            "https://jsonplaceholder.typicode.com/posts/5"
        );
        
        long parallelStart = System.currentTimeMillis();
        
        List<CompletableFuture<String>> futures = urls.stream()
            .map(url -> HttpRequest.newBuilder().uri(URI.create(url)).build())
            .map(req -> client.sendAsync(req, HttpResponse.BodyHandlers.ofString())
                .thenApply(HttpResponse::body))
            .toList();
        
        CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).join();
        
        long parallelDuration = System.currentTimeMillis() - parallelStart;
        System.out.println("    " + urls.size() + " Requests parallel: " + parallelDuration + "ms");
        
        // 3.2: Sequentiell zum Vergleich
        System.out.println("  3.2 Performance-Vergleich:");
        
        long sequentialStart = System.currentTimeMillis();
        
        for (String url : urls) {
            HttpRequest req = HttpRequest.newBuilder().uri(URI.create(url)).build();
            client.send(req, HttpResponse.BodyHandlers.ofString());
        }
        
        long sequentialDuration = System.currentTimeMillis() - sequentialStart;
        System.out.println("    " + urls.size() + " Requests sequentiell: " + sequentialDuration + "ms");
        System.out.println("    Speedup: " + String.format("%.1fx", (double) sequentialDuration / parallelDuration));
        
        // 3.3: Mit Timeout
        System.out.println("  3.3 Mit Timeout (5s):");
        
        HttpRequest timeoutRequest = HttpRequest.newBuilder()
            .uri(URI.create("https://jsonplaceholder.typicode.com/posts/1"))
            .timeout(Duration.ofSeconds(5))
            .build();
        
        try {
            HttpResponse<String> timeoutResponse = client.send(timeoutRequest, 
                HttpResponse.BodyHandlers.ofString());
            System.out.println("    Request erfolgreich: " + timeoutResponse.statusCode());
        } catch (java.net.http.HttpTimeoutException e) {
            System.out.println("    Timeout! Request abgebrochen.");
        }
        
        System.out.println();
    }
}
