# 🚀 Spring Boot Basic - Tag 3: Thymeleaf & MVC Views

**Von JSON zu HTML - Menschen-freundliche Webseiten!**

Ein vollständiges Beispielprojekt zum Spring Boot Basic Kurs von **Java Fleet Systems**.

---

## 📋 Was ist das?

Dieses Projekt zeigt dir, wie du von einer REST API (JSON) zu echten HTML-Webseiten (Views) kommst - mit **Thymeleaf** als Template Engine.

### Was du lernst:

- ✅ Unterschied zwischen `@RestController` und `@Controller`
- ✅ Thymeleaf Template Engine nutzen
- ✅ Model & View Konzept (MVC-Pattern)
- ✅ HTML-Templates mit dynamischen Daten
- ✅ Thymeleaf Expressions (`${...}`, `@{...}`)
- ✅ Schleifen mit `th:each`
- ✅ Links mit `th:href`
- ✅ **Bonus:** Pagination, Suche, Sortierung

---

## 🏗️ Projekt-Struktur

```
SpringBootBasic-Tag3/
├── pom.xml                                    # Maven Configuration
├── README.md                                  # Diese Datei
└── src/
    └── main/
        ├── java/com/javafleet/springboot/tag3/
        │   ├── SpringBootBasicTag3Application.java    # Main Class
        │   ├── controller/
        │   │   ├── HelloController.java               # Test-Controller (Tag 1)
        │   │   ├── PersonController.java              # REST API (Tag 2)
        │   │   └── PersonViewController.java          # HTML Views (Tag 3) ⭐
        │   ├── service/
        │   │   └── PersonService.java                 # Business Logic
        │   └── model/
        │       └── Person.java                        # Data Model
        └── resources/
            ├── application.properties                 # Spring Configuration
            ├── templates/                             # Thymeleaf Templates
            │   ├── persons-list.html                  # Personen-Liste ⭐
            │   └── person-details.html                # Person-Details ⭐
            └── static/
                └── css/
                    └── style.css                      # Styling ⭐
```

---

## ⚡ Schnellstart

### Voraussetzungen

- **Java 21** (oder höher)
- **Maven 3.8+**
- Eine IDE (NetBeans, IntelliJ IDEA, Eclipse, VS Code)

### 1. Projekt klonen/öffnen

```bash
# In NetBeans: File → Open Project
# Oder im Terminal:
cd SpringBootBasic-Tag3
```

### 2. Projekt starten

```bash
mvn spring-boot:run
```

Oder in NetBeans: Rechtsklick auf Projekt → Run

### 3. Im Browser öffnen

**HTML Views (für Menschen):**
- http://localhost:8080/ → Startseite (leitet zu /persons)
- http://localhost:8080/persons → Personen-Liste (Tabelle)
- http://localhost:8080/persons/1 → Person-Details

**REST API (für Programme):**
- http://localhost:8080/api/persons → Alle Personen als JSON
- http://localhost:8080/api/persons/1 → Person mit ID 1 als JSON

**Test:**
- http://localhost:8080/hello → Einfacher Test-Endpoint

---

## 🎯 Wichtige Konzepte

### 1. @RestController vs @Controller

```java
// REST API - gibt JSON zurück
@RestController
public class PersonController {
    @GetMapping("/api/persons")
    public List<Person> getAllPersons() {
        return persons;  // Spring macht daraus JSON
    }
}

// HTML Views - gibt View-Namen zurück
@Controller
public class PersonViewController {
    @GetMapping("/persons")
    public String showPersons(Model model) {
        model.addAttribute("persons", persons);
        return "persons-list";  // Name des Templates!
    }
}
```

**Faustregel:**
- API für andere Programme? → `@RestController`
- Webseite für Menschen? → `@Controller`

### 2. Model & View Konzept

```
Browser-Request
    ↓
Controller (@Controller)
    ↓
holt Daten vom Service
    ↓
packt Daten ins Model
    ↓
gibt View-Namen zurück
    ↓
Thymeleaf rendert HTML
    ↓
Browser bekommt HTML
```

### 3. Thymeleaf Expressions

```html
<!-- Variable anzeigen -->
<p th:text="${person.firstname}">Max</p>

<!-- Link erstellen -->
<a th:href="@{/persons/{id}(id=${person.id})}">Details</a>

<!-- Schleife -->
<tr th:each="person : ${persons}">
    <td th:text="${person.firstname}">Name</td>
</tr>
```

---

## 🌟 Features

### ✅ Grundlagen (CORE)

1. **Personen-Liste** (`/persons`)
   - Zeigt alle 100 Personen in einer Tabelle
   - Mit Paginierung (10 pro Seite)
   - Klickbare Spalten-Header zum Sortieren

2. **Person-Details** (`/persons/{id}`)
   - Detail-Ansicht einer einzelnen Person
   - Navigation zur vorherigen/nächsten Person
   - Link zur JSON API

3. **Suche** (`/persons?search=Anna`)
   - Suche nach Vor- oder Nachnamen
   - Live-Filterung

### 🎨 Professional

4. **Professionelles CSS**
   - Gradient-Design
   - Responsive Layout
   - Hover-Effekte
   - Moderne Farbpalette

5. **Statistiken**
   - Gesamt-Anzahl Personen
   - Aktuelle Seite
   - Personen pro Seite

### 🚀 Bonus Features

6. **Pagination**
   - Seiten-Navigation
   - Konfigurierbare Seitengröße

7. **Sortierung**
   - Nach ID, Vorname, Nachname, E-Mail
   - Aufsteigend / Absteigend
   - Visueller Indikator (↑/↓)

8. **REST API bleibt verfügbar**
   - `/api/persons` funktioniert parallel!
   - Beide nutzen denselben Service

---

## 🔧 Technologie-Stack

- **Spring Boot 3.3.5** - Framework
- **Spring Web** - REST & MVC
- **Thymeleaf** - Template Engine
- **Lombok** - Weniger Boilerplate
- **DevTools** - Hot Reload
- **Maven** - Build Tool
- **Java 21** - Programmiersprache

---

## 📚 Wichtige Endpoints

| URL | Typ | Beschreibung |
|-----|-----|--------------|
| `/` | HTML | Startseite (Redirect zu /persons) |
| `/persons` | HTML | Personen-Liste mit Tabelle |
| `/persons/{id}` | HTML | Person-Details |
| `/persons?search=Anna` | HTML | Suche nach Name |
| `/persons?page=2&size=10` | HTML | Pagination |
| `/persons?sort=firstname&order=asc` | HTML | Sortierung |
| `/api/persons` | JSON | Alle Personen (REST API) |
| `/api/persons/{id}` | JSON | Person nach ID (REST API) |
| `/hello` | TEXT | Einfacher Test |

---

## 🎓 Vergleich: Tag 1 → Tag 2 → Tag 3

### Tag 1: Erste REST API
```java
@RestController
public class HelloController {
    @GetMapping("/hello")
    public String hello() {
        return "Hello World!";
    }
}
```
→ Gibt Text zurück

### Tag 2: Spring Container & CRUD
```java
@RestController
@RequestMapping("/api/persons")
public class PersonController {
    @GetMapping
    public List<Person> getAllPersons() {
        return personService.getAllPersons();
    }
}
```
→ Gibt JSON zurück (für APIs)

### Tag 3: Thymeleaf Views (HEUTE!)
```java
@Controller
public class PersonViewController {
    @GetMapping("/persons")
    public String showPersons(Model model) {
        model.addAttribute("persons", personService.getAllPersons());
        return "persons-list";
    }
}
```
→ Gibt HTML zurück (für Menschen)

---

## 💡 Best Practices

### ✅ DO's

1. **@Controller** für HTML-Views nutzen
2. **Model** verwenden um Daten ans Template zu übergeben
3. **Thymeleaf Expressions** für dynamische Inhalte
4. **th:href** für URLs verwenden (automatischer Context-Path!)
5. **Service-Layer** wiederverwenden (REST + HTML nutzen denselben Service)

### ❌ DON'Ts

1. ❌ `@RestController` für HTML-Views
2. ❌ Direkt `new Person()` im Controller (nutze Service!)
3. ❌ Hardcoded URLs in Templates (nutze `@{...}`)
4. ❌ HTML in Java-Strings schreiben (nutze Templates!)

---

## 🐛 Troubleshooting

### Problem: Template nicht gefunden

```
Error resolving template [persons-list]
```

**Lösung:**
- Template muss in `src/main/resources/templates/` liegen
- Dateiname muss `persons-list.html` heißen (exakt wie return-Wert + `.html`)

### Problem: CSS wird nicht geladen

**Lösung:**
- CSS muss in `src/main/resources/static/css/` liegen
- Im Template: `th:href="@{/css/style.css}"` verwenden
- Spring Boot neu starten

### Problem: Thymeleaf-Syntax wird als Text angezeigt

```html
Im Browser steht: ${persons.size()}
```

**Lösung:**
- `xmlns:th="http://www.thymeleaf.org"` im `<html>` Tag hinzufügen

### Problem: 404 Error bei /persons

**Lösung:**
- `@Controller` (NICHT `@RestController`) verwenden
- Spring Boot neu starten

---

## 🔄 Was kommt als Nächstes?

### Tag 4: Thymeleaf Forms & MVC-Pattern

- Formulare mit Thymeleaf erstellen
- `th:object`, `th:field`, `th:action` verstehen
- POST Handling mit `@PostMapping`
- Validation Basics (`@Valid`, `@NotBlank`)
- Komplettes CRUD über Web-Interface

→ Dann kannst du Personen **HINZUFÜGEN** über ein Formular!

---

## 📧 Kontakt & Support

**Java Fleet Systems - Elyndra Valen**

- 📧 E-Mail: elyndra@java-developer.online
- 🌐 Website: https://java-developer.online
- 📚 Kurs: Spring Boot Basic - Tag 3

---

## 📄 Lizenz

Dieses Projekt ist Teil des **Spring Boot Basic Kurses** von Java Fleet Systems.

Frei verwendbar für Lernzwecke! 🎓

---

## 🎉 Viel Erfolg!

Du hast jetzt:
- ✅ @Controller verstanden
- ✅ Thymeleaf Templates erstellt
- ✅ Model & View Konzept gelernt
- ✅ HTML-Tabellen mit dynamischen Daten
- ✅ Links und Navigation
- ✅ Professionelles CSS

**Keep coding, keep learning!** 💙

---

_Made with ❤️ by Java Fleet Systems_
