package com.example.helloworldapi.controller;

import com.example.helloworldapi.model.Person;
import net.datafaker.Faker;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.*;

/**
 * REST Controller für Person-Daten.
 * 
 * @RequestMapping("/api/persons") setzt die Basis-URL für ALLE Endpoints in dieser Klasse.
 * Alle Methoden hier sind dann unter /api/persons/... erreichbar.
 */
@RestController
@RequestMapping("/api/persons")
public class PersonController {

    private final List<Person> persons = new ArrayList<>();

    /**
     * Konstruktor: Generiert 100 Personen mit deutschem Locale über Datafaker.
     * 
     * In einem echten Projekt kämen die Daten aus einer Datenbank.
     * Für Tag 1 reicht In-Memory – ab Tag 5 kommt die echte Datenbank!
     */
    public PersonController() {
        Faker faker = new Faker(Locale.of("de"));

        for (long i = 1; i <= 100; i++) {
            Person person = new Person(
                    i,
                    faker.name().firstName(),
                    faker.name().lastName(),
                    faker.internet().emailAddress()
            );
            persons.add(person);
        }
    }

    // =====================================================================
    // 🟢 GRUNDLAGEN: Basis-Endpoints
    // =====================================================================

    /**
     * GET /api/persons?page=0&size=10
     * 
     * Gibt alle Personen zurück – mit optionaler Paginierung.
     * Ohne Parameter: Alle 100 Personen.
     * Mit page & size: Nur den angeforderten Ausschnitt.
     * 
     * @RequestParam liest Query-Parameter aus der URL.
     * defaultValue sorgt dafür, dass ohne Parameter trotzdem alles funktioniert.
     */
    @GetMapping
    public List<Person> getAllPersons(
            @RequestParam(defaultValue = "0") int page,
            @RequestParam(defaultValue = "100") int size) {

        int start = page * size;
        int end = Math.min(start + size, persons.size());

        if (start >= persons.size()) {
            return new ArrayList<>();
        }

        return persons.subList(start, end);
    }

    /**
     * GET /api/persons/{id}
     * 
     * Gibt eine einzelne Person zurück – oder HTTP 404 wenn nicht gefunden.
     * 
     * @PathVariable extrahiert {id} aus der URL und konvertiert zu Long.
     * ResponseEntity erlaubt uns, den HTTP Status-Code zu kontrollieren.
     */
    @GetMapping("/{id}")
    public ResponseEntity<Person> getPersonById(@PathVariable Long id) {
        return persons.stream()
                .filter(p -> p.getId().equals(id))
                .findFirst()
                .map(ResponseEntity::ok)                          // 200 OK + Person als Body
                .orElse(ResponseEntity.notFound().build());       // 404 Not Found
    }

    // =====================================================================
    // 🔵 BONUS: Enterprise Features
    // =====================================================================

    /**
     * GET /api/persons/search?name=Müller
     * 
     * Sucht Personen nach Vor- oder Nachname (case-insensitive).
     * Stream API filtert die Liste und gibt alle Treffer zurück.
     */
    @GetMapping("/search")
    public List<Person> searchByName(@RequestParam String name) {
        return persons.stream()
                .filter(p -> p.getFirstname().toLowerCase().contains(name.toLowerCase())
                        || p.getLastname().toLowerCase().contains(name.toLowerCase()))
                .toList();
    }

    /**
     * GET /api/persons/count
     * 
     * Gibt die Gesamtanzahl aller Personen zurück.
     */
    @GetMapping("/count")
    public Map<String, Integer> getCount() {
        return Map.of("count", persons.size());
    }

    /**
     * GET /api/persons/random
     * 
     * Gibt eine zufällige Person zurück.
     */
    @GetMapping("/random")
    public Person getRandomPerson() {
        Random random = new Random();
        return persons.get(random.nextInt(persons.size()));
    }
}
